#pragma once

#include <solomon/services/dataproxy/lib/shard/shard_id.h>
#include <solomon/services/dataproxy/lib/shard/shard_key.h>

#include <solomon/libs/cpp/clients/memstore/stub.h>

namespace NSolomon::NDataProxy {

using TShardMap = std::unordered_map<TShardId, TShardKey>;

class TFakeMemStoreNode {
    using ListShardsRequest = yandex::monitoring::memstore::ListShardsRequest;
    using ListShardsResponse = yandex::monitoring::memstore::ListShardsResponse;

    struct TRpc: public NMemStore::TMemStoreRpcStub {
        const ui64* Generation;
        const TShardMap* Shards;

        TRpc(const ui64* generation, const TShardMap* shards)
            : Generation{generation}
            , Shards{shards}
        {
        }

        NMemStore::TListShardsAsyncResponse ListShards(const ListShardsRequest& req) override {
            ListShardsResponse resp;
            resp.set_generation(*Generation);

            if (req.generation() != *Generation) {
                for (const auto& [id, key]: *Shards) {
                    resp.add_num_ids(id);
                    auto* shardKey = resp.add_shard_keys();
                    shardKey->set_project(key.Project);
                    shardKey->set_cluster(key.SubKey.Cluster);
                    shardKey->set_service(key.SubKey.Service);
                }
            }
            return NThreading::MakeFuture(TErrorOr<decltype(resp), NGrpc::TGrpcStatus>::FromValue(resp));
        }
    };

public:
    explicit TFakeMemStoreNode(TString address)
        : Address_{std::move(address)}
    {
    }

    std::unique_ptr<NMemStore::IMemStoreRpc> MakeRpc() {
        return std::make_unique<TRpc>(&Generation_, &Shards_);
    }

    std::pair<TShardId, TShardKey> RemoveShard(TShardId id) {
        auto it = Shards_.find(id);
        Y_ENSURE(it != Shards_.end(), "cannot remove shard id=" << id);
        auto result = *it;
        Shards_.erase(it);
        Generation_++;
        return result;
    }

    void AddShard(TShardId id, TString project, TString cluster, TString service) {
        Shards_.emplace(id, TShardKey{std::move(project), std::move(cluster), std::move(service)});
        Generation_++;
    }

    const TString& Address() const noexcept {
        return Address_;
    }

private:
    TString Address_;
    TShardMap Shards_;
    ui64 Generation_{1};
};

inline auto MakeFakeRpc(const std::vector<TFakeMemStoreNode*>& nodes) {
    THashMap<TString, std::unique_ptr<NMemStore::IMemStoreRpc>> nodeRpc;
    for (auto* node: nodes) {
        nodeRpc.emplace(node->Address(), node->MakeRpc());
    }
    return std::make_shared<NMemStore::TMemStoreClusterRpcStub>(std::move(nodeRpc));
}

} // namespace NSolomon::NDataProxy
