#include <solomon/services/dataproxy/lib/memstore/shard.h>

#include <solomon/libs/cpp/actors/test_runtime/actor_runtime.h>
#include <solomon/libs/cpp/clients/memstore/stub.h>

#include <library/cpp/testing/gtest/gtest.h>

using namespace NSolomon;
using namespace NDataProxy;

using yandex::monitoring::memstore::FindRequest;
using yandex::monitoring::memstore::FindResponse;

using TFindResult = TErrorOr<FindResponse, NGrpc::TGrpcStatus>;

namespace {

class TOkNodeRpc: public NMemStore::TMemStoreRpcStub {
    NMemStore::TFindAsyncResponse Find(const FindRequest&) override {
        FindResponse resp;
        return NThreading::MakeFuture(TFindResult::FromValue(resp));
    }
};

class TErrorNodeRpc: public NMemStore::TMemStoreRpcStub {
public:
    explicit TErrorNodeRpc(grpc::StatusCode statusCode)
        : StatusCode_{statusCode}
    {
    }

    NMemStore::TFindAsyncResponse Find(const FindRequest&) override {
        return GrpcError<NMemStore::TFindAsyncResponse>(StatusCode_, "");
    }

private:
    grpc::StatusCode StatusCode_;
};

class TFailNodeRpc: public NMemStore::TMemStoreRpcStub {
    NMemStore::TFindAsyncResponse Find(const FindRequest&) override {
        return Fail<NMemStore::TFindAsyncResponse>("some fail happened");
    }
};

} // namespace

class TMemStoreShardTest: public ::testing::Test {
protected:
    void SetUp() override {
        ActorRuntime = TTestActorRuntime::CreateInited(1, false, true);
        ActorRuntime->WaitForBootstrap();
        EdgeId = ActorRuntime->AllocateEdgeActor();

        THashMap<TString, std::unique_ptr<NMemStore::IMemStoreRpc>> nodeRpc;
        nodeRpc.emplace("node-ok", std::make_unique<TOkNodeRpc>());
        nodeRpc.emplace("node-error-unavailable", std::make_unique<TErrorNodeRpc>(grpc::StatusCode::UNAVAILABLE));
        nodeRpc.emplace("node-error-aborted", std::make_unique<TErrorNodeRpc>(grpc::StatusCode::ABORTED));
        nodeRpc.emplace("node-fail", std::make_unique<TFailNodeRpc>());
        Rpc = std::make_shared<NMemStore::TMemStoreClusterRpcStub>(std::move(nodeRpc));
    }

    void TearDown() override {
        ActorRuntime.Reset();
    }

    template <typename TEvent>
    void Send(NActors::TActorId actorId, THolder<TEvent> event) {
        ActorRuntime->Send(actorId, EdgeId, std::move(event));
    }

    template <typename TEvent>
    typename TEvent::TPtr Receive() {
        return ActorRuntime->GrabEdgeEvent<TEvent>(EdgeId, TDuration::Seconds(3));
    }

public:
    THolder<TTestActorRuntime> ActorRuntime;
    NActors::TActorId EdgeId;
    std::shared_ptr<NMemStore::IMemStoreClusterRpc> Rpc;
};

TEST_F(TMemStoreShardTest, OkResponse) {
    TClusterId clusterId{EDc::Sas, EReplica::R0};
    TShardId shardId = 12345;

    auto shard = MemStoreShard(clusterId, shardId, "node-ok", Rpc);
    auto shardActorId = ActorRuntime->Register(shard.release());

    Send(shardActorId, MakeHolder<TMemStoreShardEvents::TFindReq>(FindRequest{}, TInstant::Max()));
    auto resp = Receive<TMemStoreShardEvents::TFindResp>();

    ASSERT_TRUE(resp);
    EXPECT_EQ(clusterId, resp->Get()->ClusterId);
    EXPECT_EQ(shardId, resp->Get()->ShardId);
}

TEST_F(TMemStoreShardTest, ErrorResponse) {
    TClusterId clusterId{EDc::Vla, EReplica::R1};

    {
        TShardId shardId = 3456;

        auto shard = MemStoreShard(clusterId, shardId, "node-error-unavailable", Rpc);
        auto shardActorId = ActorRuntime->Register(shard.release());

        Send(shardActorId, MakeHolder<TMemStoreShardEvents::TFindReq>(FindRequest{}, TInstant::Max()));
        auto resp = Receive<TMemStoreShardEvents::TError>();

        ASSERT_TRUE(resp);
        EXPECT_EQ(clusterId, resp->Get()->ClusterId);
        EXPECT_EQ(shardId, resp->Get()->ShardId);
        EXPECT_EQ(grpc::StatusCode::UNAVAILABLE, resp->Get()->StatusCode);
    }

    {
        TShardId shardId = 4567;

        auto shard = MemStoreShard(clusterId, shardId, "node-error-aborted", Rpc);
        auto shardActorId = ActorRuntime->Register(shard.release());

        Send(shardActorId, MakeHolder<TMemStoreShardEvents::TFindReq>(FindRequest{}, TInstant::Max()));
        auto resp = Receive<TMemStoreShardEvents::TError>();

        ASSERT_TRUE(resp);
        EXPECT_EQ(clusterId, resp->Get()->ClusterId);
        EXPECT_EQ(shardId, resp->Get()->ShardId);
        EXPECT_EQ(grpc::StatusCode::ABORTED, resp->Get()->StatusCode);
    }
}

TEST_F(TMemStoreShardTest, FailResponse) {
    TClusterId clusterId{EDc::Man, EReplica::R0};
    TShardId shardId = 1234;

    auto shard = MemStoreShard(clusterId, shardId, "node-fail", Rpc);
    auto shardActorId = ActorRuntime->Register(shard.release());

    Send(shardActorId, MakeHolder<TMemStoreShardEvents::TFindReq>(FindRequest{}, TInstant::Max()));
    auto resp = Receive<TMemStoreShardEvents::TError>();

    ASSERT_TRUE(resp);
    EXPECT_EQ(clusterId, resp->Get()->ClusterId);
    EXPECT_EQ(shardId, resp->Get()->ShardId);
    EXPECT_TRUE(resp->Get()->Message.Contains("some fail happened"));
    EXPECT_EQ(-1, resp->Get()->StatusCode);
}
