#include "cache_actor.h"
#include "message_cache.h"

#include <solomon/services/dataproxy/lib/message_cache/protos/message_cache.pb.h>
#include <solomon/services/dataproxy/config/cache_config.pb.h>

#include <solomon/libs/cpp/config/units.h>
#include <solomon/libs/cpp/http/server/handlers/metrics.h>
#include <solomon/libs/cpp/logging/logging.h>

#include <library/cpp/actors/core/actor_bootstrapped.h>
#include <library/cpp/actors/core/log.h>

#include <util/string/hex.h>
#include <util/stream/file.h>
#include <util/stream/fwd.h>

#include <google/protobuf/util/delimited_message_util.h>

using namespace NActors;
using namespace NMonitoring;

namespace NSolomon::NDataProxy {
namespace  {

class TMessageCacheActor final: public TActorBootstrapped<TMessageCacheActor> {
public:
    TMessageCacheActor(
            const TCacheConfig& config,
            std::shared_ptr<IMetricRegistry> metrics,
            ELogComponent logComponent,
            TString component,
            std::unique_ptr<IResponseFactory> responses)
        : ExpireAfter_(FromProtoTime(config.GetExpireAfter()))
        , RefreshInterval_(FromProtoTime(config.GetRefreshInterval()))
        , ProjectLimitBytes_(FromProtoDataSize(config.GetProjectLimit()))
        , Metrics_(std::move(metrics))
        , TotalMetrics_{TCacheMetrics::ForTotal(*Metrics_, component)}
        , LogComponent_{logComponent}
        , Component_{std::move(component)}
        , CachePath_{config.GetCachePath()}
        , Responses_{std::move(responses)}
    {
        // TODO: support total max size
    }

    void Bootstrap() {
        if (!CachePath_.empty() && Responses_ != nullptr) {
            LoadCache();
        }
        OnGc();
        Become(&TMessageCacheActor::StateFunc);
    }

    STATEFN(StateFunc) {
        switch (ev->GetTypeRewrite()) {
            hFunc(TCacheEvents::TLookup, OnLookup);
            hFunc(TCacheEvents::TStore, OnStore);
            sFunc(TEvents::TEvWakeup, OnGc);
            hFunc(TEvents::TEvPoison, OnPoison);
        }
    }

    void OnLookup(TCacheEvents::TLookup::TPtr& ev) {
        auto* lookup = ev->Get();
        auto key = lookup->MessageType;
        TString& project = lookup->Project;
        TString& messageHash = lookup->MessageHash;
        LOG_DEBUG_S(TActorContext::AsActorContext(), LogComponent_, '[' << __LOCATION__ << "] "
            << "OnLookup " << key << '/' << project << '/' << HexEncode(messageHash));

        auto event = std::make_unique<TCacheEvents::TLookupResult>();
        event->Project = project;
        event->MessageType = key;
        event->MessageHash = messageHash;

        if (auto it = Cache_.find(project); it != Cache_.end()) {
            TMessageCache& projectCache = it->second;
            if (auto cacheEntry = projectCache.Find(key, messageHash, TActivationContext::Now()); cacheEntry.Message) {
                event->Data = std::move(cacheEntry.Message);
                event->NeedsRefresh = cacheEntry.NeedsRefresh;
            }
        }

        LOG_DEBUG_S(TActorContext::AsActorContext(), LogComponent_, '[' << __LOCATION__ << "] "
                    "Cache " << (event->Data ? "hit " : "miss ") << key << '/' << project << '/' << HexEncode(messageHash));
        Send(ev->Sender, event.release(), 0, ev->Cookie, std::move(ev->TraceId));
    }

    void OnStore(TCacheEvents::TStore::TPtr& ev) {
        auto* store = ev->Get();
        auto key = store->MessageType;
        TString& project = store->Project;
        TString& messageHash = store->MessageHash;
        LOG_DEBUG_S(TActorContext::AsActorContext(), LogComponent_, '[' << __LOCATION__ << "] "
                    "OnStore " << key << '/' << project << '/' << HexEncode(messageHash));

        auto it = Cache_.find(project);

        if (it == Cache_.end()) {
            auto projectMetrics = TCacheMetrics::ForProject(Component_, project);
            NHttp::RegisterMetricSupplier(std::static_pointer_cast<IMetricSupplier>(projectMetrics));

            it = Cache_.emplace(std::piecewise_construct,
                                std::forward_as_tuple(project),
                                std::forward_as_tuple(
                                        ProjectLimitBytes_,
                                        ExpireAfter_,
                                        RefreshInterval_,
                                        std::move(projectMetrics),
                                        TotalMetrics_)).first;
        }

        TMessageCache& projectCache = it->second;
        projectCache.Put(key, messageHash, std::move(ev->Get()->Data), TActivationContext::Now());
    }

    void OnGc() {
        auto now = TActivationContext::Now();
        for (auto it = Cache_.begin(), end = Cache_.end(); it != end; ) {
            if (size_t evictedSize = it->second.EvictExpired(now)) {
                LOG_INFO_S(TActorContext::AsActorContext(), LogComponent_, '[' << __LOCATION__ << "] "
                           "Evicted " << evictedSize << " bytes from project " << it->first);
            }

            if (it->second.Empty()) {
                it = Cache_.erase(it);
            } else {
                ++it;
            }
        }

        Schedule(ExpireAfter_, new TEvents::TEvWakeup{});
    }

    void OnPoison(TEvents::TEvPoison::TPtr& ev) {
        if (!CachePath_.empty()) {
            SaveCache();
        }
        Send(ev->Sender, new TEvents::TEvPoisonTaken{});
        PassAway();
    }

    void LoadCache() {
        try {
            TFile file(CachePath_, RdOnly);
            if (!file.IsOpen()) {
                MON_INFO(MetaCache, "could not open a cache file: " << CachePath_);
                return;
            }
            google::protobuf::io::FileInputStream fileInput(file.GetHandle());

            bool cleanEof = false;

            while (!cleanEof) {
                yandex::monitoring::cache::Project project;
                google::protobuf::util::ParseDelimitedFromZeroCopyStream(&project, &fileInput, &cleanEof);
                if (cleanEof) {
                    return;
                }
                auto key = TString(project.name());

                auto projectMetrics = TCacheMetrics::ForProject(Component_, key);
                Cache_.emplace(std::piecewise_construct,
                               std::forward_as_tuple(key),
                               std::forward_as_tuple(ProjectLimitBytes_,
                                                     ExpireAfter_,
                                                     RefreshInterval_,
                                                     std::move(projectMetrics),
                                                     TotalMetrics_,
                                                     &fileInput,
                                                     Responses_.get(),
                                                     &cleanEof));
            }
        } catch (...) {
            MON_WARN(MetaCache, "could not load cache, " << CurrentExceptionMessage());
        }
    }

    void SaveCache() {
        try {
            TFile file(CachePath_, WrOnly);
            if (!file.IsOpen()) {
                MON_INFO(MetaCache, "could not open a cache file: " << CachePath_);
                return;
            }
            google::protobuf::io::FileOutputStream fileOutput(file.GetHandle());

            for (auto&[key, value]: Cache_) {
                yandex::monitoring::cache::Project project;
                project.set_name(key);
                google::protobuf::util::SerializeDelimitedToZeroCopyStream(project, &fileOutput);
                value.SerializeToStream(&fileOutput);
            }
        } catch (...) {
            MON_WARN(MetaCache, "could not save cache, " << CurrentExceptionMessage());
        }
    }

    static constexpr char ActorName[] = "MessageCache";

private:
    TDuration ExpireAfter_;
    TDuration RefreshInterval_;
    size_t ProjectLimitBytes_;
    std::shared_ptr<IMetricRegistry> Metrics_;
    std::shared_ptr<TCacheMetrics> TotalMetrics_;
    ELogComponent LogComponent_;
    TString Component_;
    std::unordered_map<TString, TMessageCache> Cache_;
    TString CachePath_;
    std::unique_ptr<IResponseFactory> Responses_;
};

class TMessageCacheActorStub final: public TActor<TMessageCacheActorStub> {
public:
    TMessageCacheActorStub()
        : TActor<TMessageCacheActorStub>(&TMessageCacheActorStub::StateFunc)
    {
    }

    STATEFN(StateFunc) {
        switch (ev->GetTypeRewrite()) {
            hFunc(TCacheEvents::TLookup, OnLookup);
            hFunc(TEvents::TEvPoison, OnPoison);

            IgnoreFunc(TCacheEvents::TStore);
        }
    }

    void OnLookup(TCacheEvents::TLookup::TPtr& ev) {
        auto* lookup = ev->Get();

        auto event = std::make_unique<TCacheEvents::TLookupResult>();

        event->Project = std::move(lookup->Project);
        event->MessageType = lookup->MessageType;
        event->MessageHash = std::move(lookup->MessageHash);
        event->Data = nullptr;
        event->NeedsRefresh = false;

        Send(ev->Sender, event.release(), 0, ev->Cookie, std::move(ev->TraceId));
    }

    void OnPoison(TEvents::TEvPoison::TPtr& ev) {
        Send(ev->Sender, new TEvents::TEvPoisonTaken{});
        PassAway();
    }
};

} // namespace

std::unique_ptr<IActor> CreateMessageCache(
        const TCacheConfig& config,
        std::shared_ptr<NMonitoring::IMetricRegistry> metrics,
        ELogComponent logComponent,
        TString component,
        std::unique_ptr<IResponseFactory> responses)
{
    return std::make_unique<TMessageCacheActor>(config, std::move(metrics), logComponent, std::move(component), std::move(responses));
}

std::unique_ptr<IActor> CreateMessageCacheStub() {
    return std::make_unique<TMessageCacheActorStub>();
}

} // namespace NSolomon::NDataProxy
