#include "message_cache.h"

#include <util/random/random.h>

namespace NSolomon::NDataProxy {
namespace {

TString CreateMetricPrefix(TStringBuf component) {
    static constexpr TStringBuf cachePrefix{"cache."};

    TString prefix;
    prefix.reserve(component.size() + cachePrefix.size() + 1);
    if (component) {
        prefix.append(component);
        prefix.append('.');
    }
    prefix.append(cachePrefix);
    return prefix;
}

} // namespace

void TCacheMetrics::InitMetrics(NMonitoring::IMetricRegistry& metrics, TStringBuf component) {
    auto metricPrefix = CreateMetricPrefix(component);

    HitRate = metrics.Rate(NMonitoring::MakeLabels({{"sensor", metricPrefix + "hitRate"}}));
    MissRate = metrics.Rate(NMonitoring::MakeLabels({{"sensor", metricPrefix + "missRate"}}));
    ExpireRate = metrics.Rate(NMonitoring::MakeLabels({{"sensor", metricPrefix + "expireRate"}}));
    ReplaceRate = metrics.Rate(NMonitoring::MakeLabels({{"sensor", metricPrefix + "replaceRate"}}));
    AllocationBytes = metrics.Rate(NMonitoring::MakeLabels({{"sensor", metricPrefix + "allocationBytes"}}));
    EvictionBytes = metrics.Rate(NMonitoring::MakeLabels({{"sensor", metricPrefix + "evictionBytes"}}));
    UsedBytes = metrics.IntGauge(NMonitoring::MakeLabels({{"sensor", metricPrefix + "usedBytes"}}));
}

bool TMessageCache::Put(ui32 userDefinedKey, const TString& messageHash, std::shared_ptr<const google::protobuf::Message> msg, TInstant now) {
    size_t msgSize = msg->ByteSizeLong();
    if (msgSize > LimitBytes_) {
        // do not put too big messages
        return false;
    }

    // keep cache size below maximum limit
    while (UsedBytes_ + msgSize > LimitBytes_) {
        Evict(Impl_.FindOldest());
    }

    TKey key{userDefinedKey, messageHash};
    ui64 halfRefreshIntervalMicros = RefreshInterval_.MicroSeconds() / 2;
    TInstant refreshAfter = now + TDuration::MicroSeconds(halfRefreshIntervalMicros + RandomNumber(halfRefreshIntervalMicros));
    TValue value{refreshAfter, now + ExpireAfterAccess_, std::move(msg), msgSize};

    // drop previously stored item
    if (auto it = Impl_.FindWithoutPromote(key); it != Impl_.End()) {
        Evict(it);
        Metrics_->ReplaceRate->Inc();
        TotalMetrics_->ReplaceRate->Inc();
    }

    Put(std::move(key), std::move(value));
    return true;
}

bool TMessageCache::Put(TKey key, TValue value) {
    auto msgSize = value.MessageSize;

    UsedBytes_ += msgSize;
    Impl_.Insert(std::move(key), std::move(value));

    Metrics_->AllocationBytes->Add(msgSize);
    Metrics_->UsedBytes->Add(msgSize);

    TotalMetrics_->AllocationBytes->Add(msgSize);
    TotalMetrics_->UsedBytes->Add(msgSize);
    return true;
}

TMessageCache::TCacheEntry TMessageCache::Find(ui32 userDefinedKey, TString messageHash, TInstant now) {
    TKey key{userDefinedKey, std::move(messageHash)};
    auto it = Impl_.Find(key);
    if (it == Impl_.End()) {
        Metrics_->MissRate->Inc();
        TotalMetrics_->MissRate->Inc();
        return TCacheEntry{nullptr, false};
    }

    Metrics_->HitRate->Inc();
    TotalMetrics_->HitRate->Inc();

    it->ExpireAfter = now + ExpireAfterAccess_;
    bool needsRefresh = now > it->RefreshAfter;
    if (needsRefresh) {
        Metrics_->ExpireRate->Inc();
        TotalMetrics_->ExpireRate->Inc();
    }

    return TMessageCache::TCacheEntry{it->Message, needsRefresh};
}

size_t TMessageCache::EvictExpired(TInstant now) {
    size_t evictedSize = 0;
    while (true) {
        auto it = Impl_.FindOldest();
        if (it == Impl_.End() || now < it->ExpireAfter) {
            return evictedSize;
        }

        // evict outdated item
        evictedSize += Evict(it);
    }
}

size_t TMessageCache::Evict(TLRUCache<TKey, TValue>::TIterator it) {
    size_t msgSize = it->MessageSize;
    UsedBytes_ -= msgSize;
    Impl_.Erase(it);

    Metrics_->EvictionBytes->Add(msgSize);
    TotalMetrics_->EvictionBytes->Add(msgSize);

    i64 negativeMsgSize = -1 * static_cast<i64>(msgSize);
    Metrics_->UsedBytes->Add(negativeMsgSize);
    TotalMetrics_->UsedBytes->Add(negativeMsgSize);

    return msgSize;
}

} // namespace NSolomon::NDataProxy
