#pragma once

#include <solomon/services/dataproxy/lib/message_cache/protos/message_cache.pb.h>
#include <solomon/services/dataproxy/lib/message_cache/cache_actor.h>
#include <solomon/services/dataproxy/api/dataproxy_service.pb.h>

#include <library/cpp/cache/cache.h>
#include <library/cpp/monlib/metrics/metric.h>
#include <library/cpp/monlib/metrics/metric_registry.h>
#include <library/cpp/monlib/metrics/metric_sub_registry.h>

#include <util/datetime/base.h>

#include <google/protobuf/util/delimited_message_util.h>
#include <google/protobuf/message.h>

namespace NSolomon::NDataProxy {

class TCacheMetrics: public NMonitoring::IMetricSupplier {
public:
    // should not be called directly
    TCacheMetrics() = default;

    static std::shared_ptr<TCacheMetrics> ForTotal(NMonitoring::IMetricRegistry& registry, TStringBuf component) {
        NMonitoring::TMetricSubRegistry subRegistry{{{"projectId", "total"}}, &registry};
        auto metrics = std::make_shared<TCacheMetrics>();
        metrics->InitMetrics(subRegistry, component);
        return metrics;
    }

    static std::shared_ptr<TCacheMetrics> ForProject(TStringBuf component, TStringBuf projectId) {
        auto metrics = std::make_shared<TCacheMetrics>();
        metrics->Registry_ = std::make_shared<NMonitoring::TMetricRegistry>(NMonitoring::TLabels{{"projectId", projectId}});
        metrics->InitMetrics(*metrics->Registry_, component);
        return metrics;
    }

private:
    void InitMetrics(NMonitoring::IMetricRegistry& metrics, TStringBuf component);

    void Accept(TInstant time, NMonitoring::IMetricConsumer* consumer) const override {
        if (Registry_) {
            Registry_->Accept(time, consumer);
        }
    }

    void Append(TInstant time, NMonitoring::IMetricConsumer* consumer) const override {
        if (Registry_) {
            Registry_->Append(time, consumer);
        }
    }

private:
    std::shared_ptr<NMonitoring::TMetricRegistry> Registry_;

public:
    NMonitoring::IRate* HitRate{nullptr};
    NMonitoring::IRate* MissRate{nullptr};
    NMonitoring::IRate* ExpireRate{nullptr};
    NMonitoring::IRate* ReplaceRate{nullptr};
    NMonitoring::IRate* AllocationBytes{nullptr};
    NMonitoring::IRate* EvictionBytes{nullptr};
    NMonitoring::IIntGauge* UsedBytes{nullptr};
};

class TMessageCache: private TNonCopyable {
    struct TKey {
        ui32 MessageType;
        TString MessageHash;

        bool operator<(const TKey& rhs) const noexcept {
            return MessageType < rhs.MessageType && MessageHash < rhs.MessageHash;
        }

        bool operator==(const TKey& rhs) const noexcept = default;

        explicit operator size_t() const noexcept {
            return MultiHash(MessageType, MessageHash);
        }
    };

    struct TValue {
        TInstant RefreshAfter;
        TInstant ExpireAfter;
        std::shared_ptr<const google::protobuf::Message> Message;
        size_t MessageSize{0};
    };

public:
    struct TCacheEntry {
        std::shared_ptr<const google::protobuf::Message> Message;
        bool NeedsRefresh{false};
    };

public:
    TMessageCache(
            size_t limitBytes,
            TDuration expireAfterAccess,
            TDuration refreshInterval,
            std::shared_ptr<TCacheMetrics> cacheMetrics, // per project, per shard, per-smth
            std::shared_ptr<TCacheMetrics> totalMetrics)
            : Impl_{Max<size_t>()} // do not evict by items count
            , ExpireAfterAccess_(expireAfterAccess)
            , RefreshInterval_(refreshInterval)
            , UsedBytes_{0}
            , LimitBytes_{limitBytes}
            , Metrics_{std::move(cacheMetrics)}
            , TotalMetrics_{std::move(totalMetrics)}
    {
    }

    TMessageCache(
            size_t limitBytes,
            TDuration expireAfterAccess,
            TDuration refreshInterval,
            std::shared_ptr<TCacheMetrics> cacheMetrics, // per project, per shard, per-smth
            std::shared_ptr<TCacheMetrics> totalMetrics,
            google::protobuf::io::FileInputStream* fileInput,
            const IResponseFactory* responses,
            bool* cleanEof)
        : Impl_{Max<size_t>()} // do not evict by items count
        , ExpireAfterAccess_(expireAfterAccess)
        , RefreshInterval_(refreshInterval)
        , UsedBytes_{0}
        , LimitBytes_{limitBytes}
        , Metrics_{std::move(cacheMetrics)}
        , TotalMetrics_{std::move(totalMetrics)}
    {
        auto getMessage = [&](const google::protobuf::Any& any) {
            TString key = any.type_url().substr(any.type_url().rfind('.') + 1);
            std::shared_ptr<google::protobuf::Message> mess = responses->Make(key);
            any.UnpackTo(mess.get());
            return mess;
        };
        yandex::monitoring::cache::Size size;
        google::protobuf::util::ParseDelimitedFromZeroCopyStream(&size, fileInput, cleanEof);

        for (size_t i = 0; i < size.value(); ++i) {
            yandex::monitoring::cache::Key key;
            google::protobuf::util::ParseDelimitedFromZeroCopyStream(&key, fileInput, cleanEof);

            yandex::monitoring::cache::Value value;
            google::protobuf::util::ParseDelimitedFromZeroCopyStream(&value, fileInput, cleanEof);

            Put(TKey{key.message_type(), key.message_hash()},
                TValue{
                        TInstant::MilliSeconds(value.refresh_after()),
                        TInstant::MilliSeconds(value.expire_after()),
                        getMessage(value.message()),
                        value.message_size()});
        }
    }

    bool Put(ui32 userDefinedKey, const TString& messageHash, std::shared_ptr<const google::protobuf::Message> msg, TInstant now);
    bool Put(TKey key, TValue value);
    TCacheEntry Find(ui32 userDefinedKey, TString messageHash, TInstant now);
    size_t EvictExpired(TInstant now);

    bool Empty() const {
        return Impl_.Empty();
    }

    void SerializeToStream(google::protobuf::io::FileOutputStream* fileOutput) const {
        yandex::monitoring::cache::Size size;
        size.set_value(Impl_.Size());
        google::protobuf::util::SerializeDelimitedToZeroCopyStream(size, fileOutput);
        for (auto it = Impl_.Begin(); it != Impl_.End(); ++it) {
            auto key = it.Key();
            yandex::monitoring::cache::Key protoKey;
            protoKey.set_message_type(key.MessageType);
            protoKey.set_message_hash(key.MessageHash);

            auto value = it.Value();
            yandex::monitoring::cache::Value protoValue;
            protoValue.mutable_message()->PackFrom(*value.Message);
            protoValue.set_expire_after(value.ExpireAfter.MilliSeconds());
            protoValue.set_refresh_after(value.RefreshAfter.MilliSeconds());
            protoValue.set_message_size(value.MessageSize);

            google::protobuf::util::SerializeDelimitedToZeroCopyStream(protoKey, fileOutput);
            google::protobuf::util::SerializeDelimitedToZeroCopyStream(protoValue, fileOutput);
        }
    }

private:
    size_t Evict(TLRUCache<TKey, TValue>::TIterator it);

private:
    TLRUCache<TKey, TValue> Impl_;
    TDuration ExpireAfterAccess_;
    TDuration RefreshInterval_;
    size_t UsedBytes_;
    size_t LimitBytes_;
    std::shared_ptr<TCacheMetrics> Metrics_;
    std::shared_ptr<TCacheMetrics> TotalMetrics_;
};

} // namespace NSolomon::NDataProxy
