#include <solomon/services/dataproxy/lib/api_methods/api_method.h>
#include <solomon/services/dataproxy/lib/message_cache/cache_actor.h>

#include <solomon/services/dataproxy/api/dataproxy_service.pb.h>
#include <solomon/services/dataproxy/config/cache_config.pb.h>

#include <solomon/libs/cpp/actors/test_runtime/actor_runtime.h>

#include <library/cpp/monlib/metrics/metric_registry.h>
#include <library/cpp/testing/gtest/gtest.h>

#include <library/cpp/actors/core/events.h>

#include <util/stream/file.h>

using namespace NSolomon;
using namespace NDataProxy;
using yandex::monitoring::dataproxy::FindRequest;

constexpr auto LIMIT_BYTES = 1024;
constexpr auto EXPIRE_AFTER = TDuration::Seconds(15);
constexpr auto REFRESH_INTERVAL = TDuration::Seconds(15);

struct TTestResponses : public IResponseFactory {
    std::shared_ptr<google::protobuf::Message> Make(const TString& str) const override {
        if (str == "FindRequest") {
            return std::make_shared<yandex::monitoring::dataproxy::FindRequest>();
        }
        return nullptr;
    }
};

class TCacheTest: public ::testing::Test {
protected:
    void SetUp() override {
        Metrics_ = std::make_shared<NMonitoring::TMetricRegistry>();
        ActorRuntime_ = TTestActorRuntime::CreateInited(1, false, false);

        if (auto* projectLimit = CacheConfig_.MutableProjectLimit()) {
            projectLimit->SetValue(LIMIT_BYTES);
            projectLimit->SetUnit(yandex::solomon::config::DataUnit::BYTES);
        }
        if (auto* expireAfter = CacheConfig_.MutableExpireAfter()) {
            expireAfter->SetValue(EXPIRE_AFTER.Seconds());
            expireAfter->SetUnit(yandex::solomon::config::TimeUnit::SECONDS);
        }
        if (auto* refreshInterval = CacheConfig_.MutableRefreshInterval()) {
            refreshInterval->SetValue(REFRESH_INTERVAL.Seconds());
            refreshInterval->SetUnit(yandex::solomon::config::TimeUnit::SECONDS);
        }

        TString filePath = "message_cache.cache";
        TFile file(filePath, OpenAlways);

        CacheConfig_.SetCachePath(filePath);

        ApiCache_ = ActorRuntime_->Register(
                CreateMessageCache(CacheConfig_, Metrics_, ELogComponent::UnspecifiedComponent, "", std::make_unique<TTestResponses>()));
        ActorRuntime_->WaitForBootstrap();
    }

    void TearDown() override {
        ActorRuntime_.Reset();
    }

    void Store(const TString& project, EApiMethod apiMethod, const TString& hash, const FindRequest& data) {
        auto event = std::make_unique<TCacheEvents::TStore>();
        event->Project = project;
        event->MessageType = ToUnderlying(apiMethod);
        event->MessageHash = hash;
        event->Data = std::make_unique<FindRequest>(data);
        ActorRuntime_->Send(ApiCache_, THolder(event.release()));
    }

    std::shared_ptr<const FindRequest> Lookup(const TString& project, EApiMethod apiMethod, const TString& hash) {
        auto sender = ActorRuntime_->AllocateEdgeActor();

        auto event = std::make_unique<TCacheEvents::TLookup>();
        event->Project = project;
        event->MessageType = ToUnderlying(apiMethod);
        event->MessageHash = hash;
        ActorRuntime_->Send(ApiCache_, sender, THolder(event.release()));

        auto result = ActorRuntime_->GrabEdgeEvent<TCacheEvents::TLookupResult>(sender);
        if (result->Get()->NeedsRefresh) {
            return nullptr;
        }
        return std::dynamic_pointer_cast<const FindRequest>(result->Get()->Data);
    }

    void AdvanceTime(TDuration duration) {
        ActorRuntime_->AdvanceCurrentTime(duration);
    }

    void SaveLoad() {
        auto sender = ActorRuntime_->AllocateEdgeActor();
        ActorRuntime_->Send(ApiCache_, sender, THolder(new NActors::TEvents::TEvPoison()));
        SetUp();
    }

private:
    std::shared_ptr<NMonitoring::TMetricRegistry> Metrics_;
    THolder<TTestActorRuntime> ActorRuntime_;
    NActors::TActorId ApiCache_;
    TCacheConfig CacheConfig_;
};

TEST_F(TCacheTest, CacheMiss) {
    auto fromCache = Lookup("coremon", EApiMethod::Find, "1234567890");
    ASSERT_FALSE(fromCache);
}

TEST_F(TCacheTest, CacheHit) {
    TString project = "solomon";
    EApiMethod method = EApiMethod::Find;
    TString hash = "1234567890";

    FindRequest value;
    value.set_project_id("solomon");
    value.set_limit(42);

    Store(project, method, hash, value);

    auto fromCache = Lookup(project, method, hash);
    ASSERT_TRUE(fromCache);
    EXPECT_EQ(value.project_id(), fromCache->project_id());
    EXPECT_EQ(value.limit(), fromCache->limit());
}

TEST_F(TCacheTest, SaveLoadCacheHit) {
    TString project = "solomon";
    EApiMethod method = EApiMethod::Find;
    TString hash = "1234567890";

    FindRequest value;
    value.set_project_id("solomon");
    value.set_limit(42);

    Store(project, method, hash, value);

    SaveLoad();

    auto fromCache = Lookup(project, method, hash);
    ASSERT_TRUE(fromCache);
    EXPECT_EQ(value.project_id(), fromCache->project_id());
    EXPECT_EQ(value.limit(), fromCache->limit());
}

TEST_F(TCacheTest, EvictionByTime) {
    TString project = "solomon";
    EApiMethod method = EApiMethod::Find;
    TString hash = "1234567890";

    FindRequest value;
    value.set_project_id("solomon");
    value.set_limit(42);

    Store(project, method, hash, value);

    auto fromCache1 = Lookup(project, method, hash);
    ASSERT_TRUE(fromCache1);
    EXPECT_EQ(value.project_id(), fromCache1->project_id());
    EXPECT_EQ(value.limit(), fromCache1->limit());

    AdvanceTime(EXPIRE_AFTER);

    auto fromCache2 = Lookup(project, method, hash);
    ASSERT_FALSE(fromCache2);
}

TEST_F(TCacheTest, EvictionBySize) {
    TString project = "solomon";
    EApiMethod method = EApiMethod::Find;
    TString hash = "1234567890";

    FindRequest value;
    value.set_project_id("solomon");
    value.set_limit(42);

    Store(project, method, hash, value);

    auto fromCache1 = Lookup(project, method, hash);
    ASSERT_TRUE(fromCache1);
    EXPECT_EQ(value.project_id(), fromCache1->project_id());
    EXPECT_EQ(value.limit(), fromCache1->limit());

    {
        FindRequest junk;
        junk.set_project_id(
                "Lorem ipsum dolor sit amet, consectetur adipiscing elit, sed "
                "do eiusmod tempor incididunt ut labore et dolore magna aliqua");
        junk.set_limit(42);
        junk.set_selectors(
                "Ut enim ad minim veniam, quis nostrud exercitation ullamco "
                "laboris nisi ut aliquip ex ea commodo consequat.");

        size_t times = LIMIT_BYTES / junk.ByteSizeLong() + 1;
        for (size_t i = 0; i < times; i++) {
            Store("solomon", EApiMethod::Find, ToString(i), junk);
        }
    }

    auto fromCache2 = Lookup(project, method, hash);
    ASSERT_FALSE(fromCache2);
}
