#include <solomon/services/dataproxy/lib/metabase/rpc.h>

#include <solomon/libs/cpp/grpc/status/code.h>
#include <solomon/libs/cpp/labels/known_keys.h>
#include <solomon/libs/cpp/host_resolver/host_resolver.h>
#include <solomon/libs/cpp/selectors/selectors.h>

#include <library/cpp/getopt/last_getopt.h>
#include <library/cpp/monlib/metrics/metric_registry.h>

using namespace NSolomon;
using namespace NDataProxy;
using namespace yandex::solomon::metabase;
using namespace yandex::solomon::model;
using namespace yandex::solomon::config;

constexpr ui16 METABASE_DEFAULT_PORT = 5710;

MatchType MatcherTypeToProto(EMatcherType type, bool negative) {
    switch (type) {
    case EMatcherType::EXACT: return negative ? MatchType::NOT_EXACT : MatchType::EXACT;
    case EMatcherType::REGEX: return negative ? MatchType::NOT_REGEX : MatchType::REGEX;
    case EMatcherType::ANY: return MatchType::ANY;
    case EMatcherType::ABSENT: return MatchType::ABSENT;

    case EMatcherType::GLOB:
    case EMatcherType::MULTI:
        return negative ? MatchType::NOT_GLOB : MatchType::GLOB;
    }
}

void FillMetabaseConfig(TStringBuf hosts, rpc::TGrpcClientConfig* metabaseConfig) {
    TAddressSet addresses;
    ConductorGroupResolver()->Resolve(hosts, &addresses);

    for (const auto& address: addresses) {
        if (TString::npos == address.rfind(':')) {
            metabaseConfig->add_addresses(TStringBuilder{} << address << ':' << METABASE_DEFAULT_PORT);
        } else {
            metabaseConfig->add_addresses(address);
        }
    }

    auto setTimeoutSeconds = [](auto* timeout, int value) {
        timeout->set_value(value);
        timeout->set_unit(TimeUnit::SECONDS);
    };

    setTimeoutSeconds(metabaseConfig->mutable_readtimeout(), 20);
    setTimeoutSeconds(metabaseConfig->mutable_connecttimeout(), 5);
}

int Main(const NLastGetopt::TOptsParseResult& opts) {
    bool verbose = opts.Has("verbose");
    NMonitoring::TMetricRegistry registry;

    rpc::TGrpcClientConfig metabaseConfig;
    FillMetabaseConfig(opts.Get("hosts"), &metabaseConfig);

    auto freeArgs = opts.GetFreeArgs();
    auto selectors = ParseSelectors(freeArgs[0]);
    {
        auto projectIt = selectors.Find(NLabels::LABEL_PROJECT);
        Y_ENSURE(projectIt != selectors.end(), "please provide project selector");
        Y_ENSURE(projectIt->Type() == EMatcherType::EXACT, "project selector must have exact matcher");
    }

    FindRequest request;
    for (const auto& s: selectors) {
        auto* sProto = request.add_selectors();
        sProto->set_key(TString{s.Key()});
        sProto->set_pattern(TString{s.Pattern()});
        sProto->set_match_type(MatcherTypeToProto(s.Type(), s.Negative()));
    }

    auto rpc = CreateMetabaseRpc(metabaseConfig, registry);
    const auto& addresses = rpc->Addresses();

    TVector<TAsyncFindResponse> responses;
    for (const auto& address: addresses) {
        if (verbose) {
            Cerr << "sending request to " << address << Endl;
        }
        auto* nodeRpc = rpc->Get(address);
        responses.push_back(nodeRpc->Find(request));
    }

    auto doneFuture = NThreading::WaitAll(responses);
    if (verbose) {
        Cerr << "awaiting responses ";
        while (!doneFuture.Wait(TDuration::Seconds(1))) {
            Cerr << '.';
        }
        Cerr << Endl;
    }

    for (size_t i = 0; i < responses.size(); ++i) {
        const auto& responseOrError = responses[i].GetValueSync();
        if (responseOrError.Fail()) {
            const auto& error = responseOrError.Error();
            Cerr << "got error response from " << addresses[i]
                 << " { rpcCode: " << NGrpc::StatusCodeToString(error.RpcCode)
                 << ", metabaseCode: " << EMetabaseStatusCode_Name(error.MetabaseCode)
                 << ", message: " << error.Message
                 << " }"
                 << Endl;
        } else {
            const auto& response = responseOrError.Value();
            for (const auto& m: response->metrics()) {
                Cout << MetricType_Name(m.type()) << ' ';

                Cout << '{';
                for (int j = 0; j < m.labels_size(); ++j) {
                    if (j > 0) {
                        Cout << ", ";
                    }
                    const auto& l = m.labels(j);
                    Cout << l.key() << '=' << l.value();
                }
                Cout << "} => ";
                Cout << m.metric_id().shard_id() << '/' << m.metric_id().local_id();
                Cout << '\n';
            }
        }
    }

    return 0;
}

int main(int argc, const char* argv[]) {
    NLastGetopt::TOpts opts;
    opts.SetTitle("metabase-bin");
    opts.AddLongOption("hosts", "can be single host or group of hosts prefixed with 'conductor_group://' to connect to")
            .RequiredArgument("STR")
            .Required();
    opts.AddLongOption("verbose", "print information about network actvity")
            .NoArgument()
            .Optional();
    opts.SetFreeArgsNum(1);
    opts.SetFreeArgTitle(0, "selectors");

    try {
        NLastGetopt::TOptsParseResult res(&opts, argc, argv);
        return Main(res);
    } catch (const NLastGetopt::TUsageException&) {
        opts.PrintUsage("metabase-bin");
    } catch (...) {
        Cerr << "Unhandled exception: " << CurrentExceptionMessage() << Endl;
    }

    return 1;
}
