#include "cluster.h"
#include "events.h"
#include "requester.h"
#include "shard_actor.h"
#include "watcher.h"

#include <solomon/services/dataproxy/config/cache_config.pb.h>

#include <solomon/services/dataproxy/lib/initialization/events.h>
#include <solomon/services/dataproxy/lib/message_cache/cache_actor.h>
#include <solomon/services/dataproxy/lib/message_cache/message_cache.h>
#include <solomon/services/dataproxy/lib/shard/shards_map.h>

#include <solomon/libs/cpp/config/units.h>
#include <solomon/libs/cpp/logging/logging.h>
#include <solomon/libs/cpp/selfmon/selfmon.h>

#include <library/cpp/actors/core/actor_bootstrapped.h>
#include <library/cpp/actors/core/event.h>
#include <library/cpp/actors/core/hfunc.h>

using namespace NActors;

namespace NSolomon::NDataProxy {
namespace {

class TMetabaseCluster: public TActorBootstrapped<TMetabaseCluster> {
public:
    TMetabaseCluster(
            std::vector<TString> addresses,
            IMetabaseClusterRpcPtr rpc,
            std::shared_ptr<IShardActorFactory> shardActorFactory,
            TMetabaseConfig metabaseConfig,
            TActorId cacheId,
            TActorId schedulerId)
        : Addresses_(std::move(addresses))
        , Rpc_(std::move(rpc))
        , ShardActorFactory_(std::move(shardActorFactory))
        , Config_{std::move(metabaseConfig)}
        , CleanupInterval_{FromProtoTime(Config_.GetShardsCleanupInterval())}
        , ShardTtl_{FromProtoTime(Config_.GetShardTtl())}
        , CacheId_{cacheId}
        , SchedulerId_{schedulerId}
    {
    }

    void Bootstrap() {
        WatcherId_ = Register(MetabaseClusterWatcher(Rpc_, Addresses_, TDuration::Seconds(3)).release());
        Send(WatcherId_, new TMetabaseWatcherEvents::TSubscribe);
        Become(&TThis::Normal);

        Schedule(CleanupInterval_, new TEvents::TEvWakeup);
    }

    STFUNC(Normal) {
        switch (ev->GetTypeRewrite()) {
            hFunc(TMetabaseWatcherEvents::TStateChanged, OnStateChanged);
            hFunc(TEvents::TEvWakeup, OnShardsCleanup);
            HFunc(NSelfMon::TEvPageDataReq, OnSelfMon);

            hFunc(TMetabaseEvents::TFindReq, OnRequest<TMetabaseEvents::TFindReq>);
            hFunc(TMetabaseEvents::TResolveOneReq, OnRequest<TMetabaseEvents::TResolveOneReq>);
            hFunc(TMetabaseEvents::TResolveManyReq, OnRequest<TMetabaseEvents::TResolveManyReq>);
            hFunc(TMetabaseEvents::TMetricNamesReq, OnRequest<TMetabaseEvents::TMetricNamesReq>);
            hFunc(TMetabaseEvents::TLabelNamesReq, OnRequest<TMetabaseEvents::TLabelNamesReq>);
            hFunc(TMetabaseEvents::TLabelValuesReq, OnRequest<TMetabaseEvents::TLabelValuesReq>);
            hFunc(TMetabaseEvents::TUniqueLabelsReq, OnRequest<TMetabaseEvents::TUniqueLabelsReq>);

            HFunc(TInitializationEvents::TSubscribe, OnInitializationSubscribe);

            hFunc(TEvents::TEvPoison, OnPoison);
        }
    }

    STATEFN(Dying) {
        switch (ev->GetTypeRewrite()) {
            sFunc(TEvents::TEvPoison, OnHalt);
            sFunc(TEvents::TEvPoisonTaken, OnPoisonTaken);
        }
    }

private:
    void OnStateChanged(TMetabaseWatcherEvents::TStateChanged::TPtr& ev) {
        for (const auto& shardInfo: ev->Get()->Updated) {
            if (ShardsMap_.Update(shardInfo)) {
                auto it = ShardIdToShardState_.find(shardInfo->Id);

                if (it == ShardIdToShardState_.end()) {
                    continue;
                }

                Send(it->second.ActorId, new TMetabaseShardActorEvents::TShardUpdate{shardInfo});
            }
        }

        for (const auto& id: ev->Get()->Removed) {
            if (auto shardInfo = ShardsMap_.Remove(id)) {
                const auto actorIt = ShardIdToShardState_.find(shardInfo->Id);

                if (actorIt == ShardIdToShardState_.end()) {
                    continue;
                }

                Send(actorIt->second.ActorId, new TEvents::TEvPoison);
                ShardIdToShardState_.erase(actorIt);
            }
        }
    }

    void OnShardsCleanup(TEvents::TEvWakeup::TPtr& ev) {
        auto now = TActivationContext::Now();

        for (auto it = ShardIdToShardState_.begin(), end = ShardIdToShardState_.end(); it != end;) {
            const auto& shardState = it->second;

            if (now - shardState.LastAccessedAt > ShardTtl_) {
                auto toDelete = it++;

                Send(toDelete->second.ActorId, new TEvents::TEvPoison);
                ShardIdToShardState_.erase(toDelete);
            } else {
                ++it;
            }
        }

        Schedule(CleanupInterval_, ev->Release().Release());
    }

    TActorId GetOrCreateShardActor(const TString& project, const TShardLocation& shardLoc) {
        auto& shardState = ShardIdToShardState_[shardLoc.Id];
        shardState.LastAccessedAt = TActivationContext::Now();

        if (!shardState.ActorId) {
            // TODO(ivanzhukov): destroy an actor if there were no requests for too long
            auto maxInflight = Config_.GetRequestInflightPerShard();
            auto actor = ShardActorFactory_->Create(project, Rpc_, shardLoc, maxInflight, CacheId_);
            shardState.ActorId = Register(actor.release());
        }

        return shardState.ActorId;
    }

    template <typename TReq>
    void OnRequest(typename TReq::TPtr& ev) {
        const auto& message = ev->Get()->Message;
        const TShardSelector& shardSelector = ev->Get()->ShardSelector;
        MON_DEBUG(MetabaseClient, message->GetTypeName() << " {" << message->ShortDebugString() << '}');

        TVector<TShardActorId> shardActors;

        if (shardSelector.IsExact()) {
            if (auto shard = ShardsMap_.FindExact(shardSelector)) {
                const auto& shardLocation = shard.value();
                shardActors.emplace_back(TShardActorId{
                    shardLocation.Id,
                    GetOrCreateShardActor(shardSelector.Project, shard.value()),
                });
            }
        } else if (auto shards = ShardsMap_.Find(shardSelector); !shards.empty()) {
            for (const auto& shardLocation: shards) {
                shardActors.emplace_back(TShardActorId{
                    shardLocation.Id,
                    GetOrCreateShardActor(shardSelector.Project, shardLocation),
                });
            }
        }

        if (!shardActors.empty()) {
            auto requester = ShardsRequester<TReq>(std::move(shardActors), SchedulerId_);
            auto requesterId = Register(requester.release());
            TActivationContext::Send(ev->Forward(requesterId));

            return;
        }

        MON_WARN(MetabaseClient, "cannot find shard(s) by selector " << shardSelector);
        auto event = std::make_unique<TMetabaseEvents::TError>();
        event->RpcCode = grpc::StatusCode::UNKNOWN;
        event->MetabaseCode = yandex::solomon::metabase::EMetabaseStatusCode::SHARD_NOT_FOUND;
        event->Message = TStringBuilder{} << "shard(s) not found by selector " << shardSelector;
        this->Send(ev->Sender, event.release(), 0, ev->Cookie);
        this->Send(ev->Sender, new TMetabaseEvents::TDone, 0, ev->Cookie, std::move(ev->TraceId));
    }

    void OnInitializationSubscribe(TInitializationEvents::TSubscribe::TPtr& ev, const TActorContext& ctx) {
        ctx.Send(ev->Forward(WatcherId_));
    }

    void OnPoison(TEvents::TEvPoison::TPtr& ev) {
        PoisonerId_ = ev->Sender;
        Become(&TThis::Dying);

        if (!ShardIdToShardState_.empty()) {
            NeedToBePoisoned_ += ShardIdToShardState_.size();

            for (const auto [_, shardState]: ShardIdToShardState_) {
                Send(shardState.ActorId, new TEvents::TEvPoison);
            }
        }

        if (CacheId_) {
            ++NeedToBePoisoned_;
            Send(CacheId_, new TEvents::TEvPoison);
        }

        if (WatcherId_) {
            ++NeedToBePoisoned_;
            Send(WatcherId_, new TEvents::TEvPoison);
        }

        if (NeedToBePoisoned_ == 0) {
            OnHalt();
            return;
        }
    }

    void OnPoisonTaken() {
        if (--NeedToBePoisoned_ == 0) {
            OnHalt();
        }
    }

    void OnSelfMon(const NSelfMon::TEvPageDataReq::TPtr& ev, const TActorContext& ctx) {
        ctx.Send(ev->Forward(WatcherId_));
    }

    void OnHalt() {
        if (PoisonerId_) {
            Send(PoisonerId_, new TEvents::TEvPoisonTaken);
        }

        PassAway();
    }

private:
    std::vector<TString> Addresses_;
    IMetabaseClusterRpcPtr Rpc_;
    std::shared_ptr<IShardActorFactory> ShardActorFactory_;
    TMetabaseConfig Config_;
    TDuration CleanupInterval_;
    TDuration ShardTtl_;
    TActorId CacheId_;
    TActorId SchedulerId_;

    TActorId WatcherId_;
    TShardsMap ShardsMap_;
    TActorId PoisonerId_;
    size_t NeedToBePoisoned_{0};

    struct TShardState {
        TActorId ActorId;
        TInstant LastAccessedAt;
    };

    std::unordered_map<TShardId, TShardState> ShardIdToShardState_;
};

} // namespace

std::unique_ptr<IActor> MetabaseCluster(
        std::vector<TString> addresses,
        IMetabaseClusterRpcPtr rpc,
        std::shared_ptr<IShardActorFactory> shardActorFactory,
        TMetabaseConfig metabaseConfig,
        TActorId cacheId,
        TActorId schedulerId)
{
    return std::make_unique<TMetabaseCluster>(
            std::move(addresses),
            std::move(rpc),
            std::move(shardActorFactory),
            std::move(metabaseConfig),
            std::move(cacheId),
            std::move(schedulerId));
}

} // namespace NSolomon::NDataProxy
