#pragma once

#include <solomon/services/dataproxy/lib/event_slots.h>
#include <solomon/services/dataproxy/lib/selectors/shard_selector.h>

#include <solomon/libs/cpp/actors/events/events.h>
#include <solomon/protos/metabase/grpc_create.pb.h>
#include <solomon/protos/metabase/grpc_delete.pb.h>
#include <solomon/protos/metabase/grpc_find.pb.h>
#include <solomon/protos/metabase/grpc_label_names.pb.h>
#include <solomon/protos/metabase/grpc_label_unique.pb.h>
#include <solomon/protos/metabase/grpc_label_values.pb.h>
#include <solomon/protos/metabase/grpc_resolve.pb.h>
#include <solomon/protos/metabase/grpc_metric_names.pb.h>
#include <solomon/protos/metabase/grpc_status.pb.h>

#include <library/cpp/actors/core/event_local.h>

#include <util/generic/typelist.h>

#include <grpc++/support/status_code_enum.h>

namespace NSolomon::NDataProxy {

class TMetabaseEvents: private TEventSlot<EEventSpace::DataProxy, ES_METABASE_CLUSTER> {

    template <typename TDerived, typename TMessage, ui32 Type>
    struct TRequest: public NActors::TEventLocal<TDerived, Type> {
        using TProtoMsg = TMessage;

        TShardSelector ShardSelector;
        std::shared_ptr<const TMessage> Message;
        TInstant Deadline;

        TRequest() = default;

        TRequest(const TRequest& rhs)
            : ShardSelector{rhs.ShardSelector}
            , Message{rhs.Message}
            , Deadline{rhs.Deadline}
        {
        }

        TRequest(TRequest&& rhs)
            : ShardSelector{std::move(rhs.ShardSelector)}
            , Message{std::move(rhs.Message)}
            , Deadline{rhs.Deadline}
        {
        }
    };

    template <typename TDerived, typename TMessage, ui32 Type>
    struct TResponse: public NActors::TEventLocal<TDerived, Type> {
        using TProtoMsg = TMessage;

        std::shared_ptr<const TMessage> Message;
    };

    enum {
        FindReq = SpaceBegin,
        FindResp,
        ResolveOneReq,
        ResolveOneResp,
        ResolveManyReq,
        ResolveManyResp,
        MetricNamesReq,
        MetricNamesResp,
        LabelValuesReq,
        LabelValuesResp,
        LabelNamesReq,
        LabelNamesResp,
        UniqueLabelsReq,
        UniqueLabelsResp,
        Error,
        Wakeup,
        Done,
        End,
    };
    static_assert(End < SpaceEnd, "too many event types");

public:
    struct TFindReq: public TRequest<TFindReq, yandex::solomon::metabase::FindRequest, FindReq>{};
    struct TFindResp: public TResponse<TFindResp, yandex::solomon::metabase::FindResponse, FindResp>{};

    struct TResolveOneReq: public TRequest<TResolveOneReq, yandex::solomon::metabase::ResolveOneRequest, ResolveOneReq>{};
    struct TResolveOneResp: public TResponse<TResolveOneResp, yandex::solomon::metabase::ResolveOneResponse, ResolveOneResp>{};

    struct TResolveManyReq: public TRequest<TResolveManyReq, yandex::solomon::metabase::ResolveManyRequest, ResolveManyReq>{};
    struct TResolveManyResp: public TResponse<TResolveManyResp, yandex::solomon::metabase::ResolveManyResponse, ResolveManyResp>{};

    struct TMetricNamesReq: public TRequest<TMetricNamesReq, yandex::solomon::metabase::MetricNamesRequest, MetricNamesReq>{};
    struct TMetricNamesResp: public TResponse<TMetricNamesResp, yandex::solomon::metabase::MetricNamesResponse, MetricNamesResp>{};

    struct TLabelValuesReq: public TRequest<TLabelValuesReq, yandex::solomon::metabase::TLabelValuesRequest, LabelValuesReq>{};
    struct TLabelValuesResp: public TResponse<TLabelValuesResp, yandex::solomon::metabase::TLabelValuesResponse, LabelValuesResp>{};

    struct TLabelNamesReq: public TRequest<TLabelNamesReq, yandex::solomon::metabase::TLabelNamesRequest, LabelNamesReq>{};
    struct TLabelNamesResp: public TResponse<TLabelNamesResp, yandex::solomon::metabase::TLabelNamesResponse, LabelNamesResp>{};

    struct TUniqueLabelsReq: public TRequest<TUniqueLabelsReq, yandex::solomon::metabase::TUniqueLabelsRequest, UniqueLabelsReq>{};
    struct TUniqueLabelsResp: public TResponse<TUniqueLabelsResp, yandex::solomon::metabase::TUniqueLabelsResponse, UniqueLabelsResp>{};

    struct TWakeupEvent: public NActors::TEventLocal<TWakeupEvent, Wakeup>{};

    using TListOfReqAndResp = TTypeList<
            std::pair<TFindReq, TFindResp>,
            std::pair<TResolveOneReq, TResolveOneResp>,
            std::pair<TResolveManyReq, TResolveManyResp>,
            std::pair<TMetricNamesReq, TMetricNamesResp>,
            std::pair<TLabelValuesReq, TLabelValuesResp>,
            std::pair<TLabelNamesReq, TLabelNamesResp>,
            std::pair<TUniqueLabelsReq, TUniqueLabelsResp>
            >;

    template <typename TReq>
    struct TIsKnownReq {
        template <typename TReqAndResp>
        using TResult = TBoolConstant<std::is_same_v<TReq, typename TReqAndResp::first_type>>;
    };

    template <typename TResp>
    struct TIsKnownResp {
        template <typename TReqAndResp>
        using TResult = TBoolConstant<std::is_same_v<TResp, typename TReqAndResp::second_type>>;
    };

    template <typename TReq>
    struct TRequestToResponse {
        using TResponse = typename TListOfReqAndResp::TSelectBy<TIsKnownReq<TReq>::template TResult>::type::second_type;
    };

    template <typename TResp>
    struct TResponseToRequest {
        using TRequest = typename TListOfReqAndResp::TSelectBy<TIsKnownResp<TResp>::template TResult>::type::first_type;
    };

    struct TError: public NActors::TEventLocal<TError, Error> {
        grpc::StatusCode RpcCode;
        yandex::solomon::metabase::EMetabaseStatusCode MetabaseCode;
        TString Message;
    };

    struct TDone: public NActors::TEventLocal<TDone, Done> {
    };
};

} // namespace NSolomon::NDataProxy
