#include "rpc.h"

#include <solomon/libs/cpp/grpc/status/code.h>
#include <solomon/protos/metabase/metabase_service.grpc.pb.h>

#include <library/cpp/grpc/client/grpc_client_low.h>
#include <library/cpp/monlib/metrics/metric_registry.h>

namespace NSolomon::NDataProxy {
namespace {

using namespace NThreading;
using namespace NGrpc;
using namespace yandex::solomon::metabase;

using yandex::solomon::config::rpc::TGrpcClientConfig;
using yandex::solomon::config::DataUnit;
using yandex::solomon::config::TimeUnit;
using yandex::monitoring::metabase::MetabaseService;

/**
 * Actual metabse RPC implementation.
 */
class TMetabaseRpcImpl final: public IMetabaseRpc {
    using TStub = MetabaseService::Stub;

public:
    TMetabaseRpcImpl(
            NMonitoring::IMetricRegistry&,
            std::unique_ptr<TGrpcServiceConnection<MetabaseService>> connection)
        : Connection_{std::move(connection)}
    {
    }

    TAsyncMetabaseStatusResponse ServerStatus(const TServerStatusRequest& request) override {
        return SendRequest<TServerStatusRequest, TServerStatusResponse>(request, &TStub::AsyncServerStatus);
    }

    TAsyncFindResponse Find(const FindRequest& request) override {
        return SendRequest<FindRequest, FindResponse>(request, &TStub::AsyncFind);
    }

    TAsyncCreateOneResponse CreateOne(const CreateOneRequest& request) override {
        return SendRequest<CreateOneRequest, CreateOneResponse>(request, &TStub::AsyncCreateOne);
    }

    TAsyncCreateManyResponse CreateMany(const CreateManyRequest& request) override {
        return SendRequest<CreateManyRequest, CreateManyResponse>(request, &TStub::AsyncCreateMany);
    }

    TAsyncResolveOneResponse ResolveOne(const ResolveOneRequest& request) override {
        return SendRequest<ResolveOneRequest, ResolveOneResponse>(request, &TStub::AsyncResolveOne);
    }

    TAsyncResolveManyResponse ResolveMany(const ResolveManyRequest& request) override {
        return SendRequest<ResolveManyRequest, ResolveManyResponse>(request, &TStub::AsyncResolveMany);
    }

    TAsyncDeleteManyResponse DeleteMany(const DeleteManyRequest& request) override {
        return SendRequest<DeleteManyRequest, DeleteManyResponse>(request, &TStub::AsyncDeleteMany);
    }

    TAsyncMetricNamesResponse MetricNames(const MetricNamesRequest& request) override {
        return SendRequest<MetricNamesRequest, MetricNamesResponse>(request, &TStub::AsyncMetricNames);
    }

    TAsyncLabelValuesResponse LabelValues(const TLabelValuesRequest& request) override {
        return SendRequest<TLabelValuesRequest, TLabelValuesResponse>(request, &TStub::AsyncLabelValues);
    }

    TAsyncLabelNamesResponse LabelNames(const TLabelNamesRequest& request) override {
        return SendRequest<TLabelNamesRequest, TLabelNamesResponse>(request, &TStub::AsyncLabelNames);
    }

    TAsyncUniqueLabelsResponse UniqueLabels(const TUniqueLabelsRequest& request) override {
        return SendRequest<TUniqueLabelsRequest, TUniqueLabelsResponse>(request, &TStub::AsyncUniqueLabels);
    }

private:
    template <typename TReq, typename TResp, typename TProcessor = TSimpleRequestProcessor<MetabaseService::Stub, TReq, TResp>>
    TFuture<TErrorOr<std::unique_ptr<const TResp>, TMetabaseError>> SendRequest(const TReq& req, typename TProcessor::TAsyncRequest asyncRequest) {
        using TResult = TErrorOr<std::unique_ptr<const TResp>, TMetabaseError>;

        auto promise = NThreading::NewPromise<TResult>();
        Connection_->Request<TReq, TResp>(
            req,
            [promise](TGrpcStatus&& status, TResp&& resp) mutable {
                if (status.Ok()) {
                    if (resp.GetStatus() == EMetabaseStatusCode::OK) {
                        promise.SetValue(std::make_unique<const TResp>(std::move(resp)));
                    } else {
                        promise.SetValue(TMetabaseError{grpc::StatusCode::OK, resp.GetStatus(), resp.GetStatusMessage()});
                    }
                } else {
                    auto rpcCode = static_cast<grpc::StatusCode>(status.GRpcStatusCode);
                    promise.SetValue(TMetabaseError{rpcCode, EMetabaseStatusCode::UNKNOWN, status.Msg});
                }
            },
            asyncRequest,
            CreateCallMeta(TInstant::MilliSeconds(req.GetDeadlineMillis())));
        return promise.GetFuture();
    }

    TCallMeta CreateCallMeta(TInstant deadline) {
        TCallMeta meta;
        if (deadline) {
            meta.Timeout = deadline;
        } else {
            meta.Timeout = Connection_->Timeout();
        }
        return meta;
    }

private:
    std::unique_ptr<TGrpcServiceConnection<MetabaseService>> Connection_;
};

IMetabaseRpcPtr CreateMetabaseClient(
        NMonitoring::IMetricRegistry& registry,
        std::unique_ptr<TGrpcServiceConnection<MetabaseService>> connection)
{
    return std::make_unique<TMetabaseRpcImpl>(registry, std::move(connection));
}

TGrpcClientConfig CreateConfig(const TString& address) {
    TGrpcClientConfig conf;
    conf.AddAddresses(address);
    conf.SetWorkerThreads(2);
    {
        auto* inMaxSize = conf.MutableMaxInboundMessageSize();
        inMaxSize->SetValue(16);
        inMaxSize->SetUnit(DataUnit::MEGABYTES);
    }
    {
        auto* outMaxSize = conf.MutableMaxOutboundMessageSize();
        outMaxSize->SetValue(1);
        outMaxSize->SetUnit(DataUnit::MEGABYTES);
    }
    {
        auto* connectTimeout = conf.MutableConnectTimeout();
        connectTimeout->SetValue(5);
        connectTimeout->SetUnit(TimeUnit::SECONDS);
    }
    {
        auto* readTimeout = conf.MutableReadTimeout();
        readTimeout->SetValue(25);
        readTimeout->SetUnit(TimeUnit::SECONDS);
    }
    return conf;
}

} // namespace

void TMetabaseError::Out(IOutputStream& out) const {
    out << "{ rpcCode: " << NGrpc::StatusCodeToString(RpcCode);
    out << ", metabaseCode: " << EMetabaseStatusCode_Name(MetabaseCode);

    if (Message.empty()) {
        out << " }";
    } else {
        out << ", message: '" << Message << "' }";
    }
}

/**
 * Single node metabse RPC.
 */
IMetabaseRpcPtr CreateMetabaseRpc(const TString& address, NMonitoring::TMetricRegistry& registry, TString clientId) {
    auto conf = CreateConfig(address);
    auto threadPool = CreateGrpcThreadPool(conf);
    auto sc = CreateGrpcServiceConnection<MetabaseService>(
        conf,
        false,
        registry,
        std::move(threadPool),
        std::move(clientId));

    return CreateMetabaseClient(registry, std::move(sc));
}

/**
 * Cluster aware metabase RPC.
 */
IMetabaseClusterRpcPtr CreateMetabaseRpc(
        const yandex::solomon::config::rpc::TGrpcClientConfig& conf,
        NMonitoring::TMetricRegistry& registry,
        TString clientId)
{
    return std::make_shared<TGrpcClusterClientBase<MetabaseService, IMetabaseRpc>>(
        conf,
        false,
        registry,
        CreateMetabaseClient,
        std::move(clientId));
}

} // namespace NSolomon::NDataProxy

template <>
void Out<NSolomon::NDataProxy::TMetabaseError>(IOutputStream& o, const NSolomon::NDataProxy::TMetabaseError& e) {
    e.Out(o);
}
