#pragma once

#include <solomon/libs/cpp/error_or/error_or.h>
#include <solomon/libs/cpp/grpc/client/client.h>

#include <solomon/protos/configs/rpc/rpc_config.pb.h>
#include <solomon/protos/metabase/grpc_find.pb.h>
#include <solomon/protos/metabase/grpc_status.pb.h>
#include <solomon/protos/metabase/grpc_create.pb.h>
#include <solomon/protos/metabase/grpc_resolve.pb.h>
#include <solomon/protos/metabase/grpc_delete.pb.h>
#include <solomon/protos/metabase/grpc_metric_names.pb.h>
#include <solomon/protos/metabase/grpc_label_names.pb.h>
#include <solomon/protos/metabase/grpc_label_values.pb.h>
#include <solomon/protos/metabase/grpc_label_unique.pb.h>
#include <solomon/protos/metabase/status_code.pb.h>

#include <library/cpp/monlib/metrics/fwd.h>
#include <library/cpp/threading/future/future.h>

#include <grpc++/support/status_code_enum.h>

namespace NSolomon::NDataProxy {

/**
 * RPC or Metabase error.
 */
struct TMetabaseError {
    grpc::StatusCode RpcCode;
    yandex::solomon::metabase::EMetabaseStatusCode MetabaseCode;
    TString Message;

    TMetabaseError(
            grpc::StatusCode rpcCode,
            yandex::solomon::metabase::EMetabaseStatusCode metabaseCode,
            TString message) noexcept
        : RpcCode{rpcCode}
        , MetabaseCode{metabaseCode}
        , Message{std::move(message)}
    {
    }

    explicit TMetabaseError(TString message) noexcept
        : RpcCode{grpc::StatusCode::UNKNOWN}
        , MetabaseCode{yandex::solomon::metabase::EMetabaseStatusCode::UNKNOWN}
        , Message{std::move(message)}
    {
    }

    void Out(IOutputStream& out) const;
};

inline bool IsRetryableError(yandex::solomon::metabase::EMetabaseStatusCode code) noexcept {
    switch (code) {
        case yandex::solomon::metabase::EMetabaseStatusCode::NOT_FOUND:
        case yandex::solomon::metabase::EMetabaseStatusCode::NODE_UNAVAILABLE:
        case yandex::solomon::metabase::EMetabaseStatusCode::SHARD_NOT_FOUND:
        case yandex::solomon::metabase::EMetabaseStatusCode::SHARD_NOT_READY:
        case yandex::solomon::metabase::EMetabaseStatusCode::RESOURCE_EXHAUSTED:
            return true;

        default:
            return false;
    }
}

template <typename TResp>
using TMetabaseAsyncResponse = NThreading::TFuture<TErrorOr<std::unique_ptr<const TResp>, TMetabaseError>>;

using TAsyncFindResponse = TMetabaseAsyncResponse<yandex::solomon::metabase::FindResponse>;
using TAsyncMetabaseStatusResponse = TMetabaseAsyncResponse<yandex::solomon::metabase::TServerStatusResponse>;
using TAsyncCreateOneResponse = TMetabaseAsyncResponse<yandex::solomon::metabase::CreateOneResponse>;
using TAsyncCreateManyResponse = TMetabaseAsyncResponse<yandex::solomon::metabase::CreateManyResponse>;
using TAsyncResolveOneResponse = TMetabaseAsyncResponse<yandex::solomon::metabase::ResolveOneResponse>;
using TAsyncResolveManyResponse = TMetabaseAsyncResponse<yandex::solomon::metabase::ResolveManyResponse>;
using TAsyncDeleteManyResponse = TMetabaseAsyncResponse<yandex::solomon::metabase::DeleteManyResponse>;
using TAsyncMetricNamesResponse = TMetabaseAsyncResponse<yandex::solomon::metabase::MetricNamesResponse>;
using TAsyncLabelValuesResponse = TMetabaseAsyncResponse<yandex::solomon::metabase::TLabelValuesResponse>;
using TAsyncLabelNamesResponse = TMetabaseAsyncResponse<yandex::solomon::metabase::TLabelNamesResponse>;
using TAsyncUniqueLabelsResponse = TMetabaseAsyncResponse<yandex::solomon::metabase::TUniqueLabelsResponse>;

/**
 * Interface of single node metabase RPC.
 */
class IMetabaseRpc: public TThrRefBase {
public:
    // Fetches metada about particular node. Response will contains shards that located
    // on this node and another helpful information
    virtual TAsyncMetabaseStatusResponse ServerStatus(const yandex::solomon::metabase::TServerStatusRequest& request) = 0;

    // Finds metric ids by specified selectors in particular node. Each node have only part of whole data
    virtual TAsyncFindResponse Find(const yandex::solomon::metabase::FindRequest& request) = 0;

    // Creates single metric with specified settings
    virtual TAsyncCreateOneResponse CreateOne(const yandex::solomon::metabase::CreateOneRequest& request) = 0;

    // Creates multiple metrics with specified settings
    virtual TAsyncCreateManyResponse CreateMany(const yandex::solomon::metabase::CreateManyRequest& request) = 0;

    // Resolves metric id by unique label list sequence
    virtual TAsyncResolveOneResponse ResolveOne(const yandex::solomon::metabase::ResolveOneRequest& request) = 0;

    // Resolves multiple metrics by unique label list sequences
    virtual TAsyncResolveManyResponse ResolveMany(const yandex::solomon::metabase::ResolveManyRequest& request) = 0;

    // Deletes multiple metrics
    virtual TAsyncDeleteManyResponse DeleteMany(const yandex::solomon::metabase::DeleteManyRequest& request) = 0;

    // Finds metric names by specified selectors
    virtual TAsyncMetricNamesResponse MetricNames(const yandex::solomon::metabase::MetricNamesRequest& request) = 0;

    // Resolves available values for labels by specified selector
    virtual TAsyncLabelValuesResponse LabelValues(const yandex::solomon::metabase::TLabelValuesRequest& request) = 0;

    // Resolves available names for labels by specified selector
    virtual TAsyncLabelNamesResponse LabelNames(const yandex::solomon::metabase::TLabelNamesRequest& request) = 0;

    // Resolves metrics by specified selector and find unique label pairs
    virtual TAsyncUniqueLabelsResponse UniqueLabels(const yandex::solomon::metabase::TUniqueLabelsRequest& request) = 0;
};
using IMetabaseRpcPtr = std::unique_ptr<IMetabaseRpc>;
using IMetabaseClusterRpc = IClusterRpc<IMetabaseRpc>;
using IMetabaseClusterRpcPtr = std::shared_ptr<IMetabaseClusterRpc>;

/**
 * Creates single node metabase RPC.
 */
IMetabaseRpcPtr CreateMetabaseRpc(const TString& address, NMonitoring::TMetricRegistry& registry, TString clientId = {});

/**
 * Creates metabase RPC for each cluster node.
 */
IMetabaseClusterRpcPtr CreateMetabaseRpc(
        const yandex::solomon::config::rpc::TGrpcClientConfig& conf,
        NMonitoring::TMetricRegistry& registry,
        TString clientId = {});

} // namespace NSolomon::NDataProxy
