#include "rpc_stub.h"

#include <util/generic/hash.h>

namespace NSolomon::NDataProxy {
namespace {

using namespace yandex::solomon::metabase;

/**
 * Stub single node RPC implementation.
 */
class TStubMetabaseRpc: public IMetabaseRpc {
public:
    explicit TStubMetabaseRpc(TStubMetabaseHandlers handlers)
        : Handlers_{std::move(handlers)}
    {
    }

    TAsyncMetabaseStatusResponse ServerStatus(const TServerStatusRequest& req) override {
        if (Handlers_.OnServerStatus) {
            return Handlers_.OnServerStatus(req);
        }
        return MetabaseFailedResponse<TAsyncMetabaseStatusResponse>("not implemented");
    }

    TAsyncFindResponse Find(const FindRequest& req) override {
        if (Handlers_.OnFind) {
            return Handlers_.OnFind(req);
        }
        return MetabaseFailedResponse<TAsyncFindResponse>("not implemented");
    }

    TAsyncCreateOneResponse CreateOne(const CreateOneRequest& req) override {
        if (Handlers_.OnCreateOne) {
            return Handlers_.OnCreateOne(req);
        }
        return MetabaseFailedResponse<TAsyncCreateOneResponse>("not implemented");
    }

    TAsyncCreateManyResponse CreateMany(const CreateManyRequest& req) override {
        if (Handlers_.OnCreateMany) {
            return Handlers_.OnCreateMany(req);
        }
        return MetabaseFailedResponse<TAsyncCreateManyResponse>("not implemented");
    }

    TAsyncResolveOneResponse ResolveOne(const ResolveOneRequest& req) override {
        if (Handlers_.OnResolveOne) {
            return Handlers_.OnResolveOne(req);
        }
        return MetabaseFailedResponse<TAsyncResolveOneResponse>("not implemented");
    }

    TAsyncResolveManyResponse ResolveMany(const ResolveManyRequest& req) override {
        if (Handlers_.OnResolveMany) {
            return Handlers_.OnResolveMany(req);
        }
        return MetabaseFailedResponse<TAsyncResolveManyResponse>("not implemented");
    }

    TAsyncDeleteManyResponse DeleteMany(const DeleteManyRequest& req) override {
        if (Handlers_.OnDeleteMany) {
            return Handlers_.OnDeleteMany(req);
        }
        return MetabaseFailedResponse<TAsyncDeleteManyResponse>("not implemented");
    }

    TAsyncMetricNamesResponse MetricNames(const MetricNamesRequest& req) override {
        if (Handlers_.OnMetricNames) {
            return Handlers_.OnMetricNames(req);
        }
        return MetabaseFailedResponse<TAsyncMetricNamesResponse>("not implemented");
    }

    TAsyncLabelValuesResponse LabelValues(const TLabelValuesRequest& req) override {
        if (Handlers_.OnLabelValues) {
            return Handlers_.OnLabelValues(req);
        }
        return MetabaseFailedResponse<TAsyncLabelValuesResponse>("not implemented");
    }

    TAsyncLabelNamesResponse LabelNames(const TLabelNamesRequest& req) override {
        if (Handlers_.OnLabelNames) {
            return Handlers_.OnLabelNames(req);
        }
        return MetabaseFailedResponse<TAsyncLabelNamesResponse>("not implemented");
    }

    TAsyncUniqueLabelsResponse UniqueLabels(const TUniqueLabelsRequest& req) override {
        if (Handlers_.OnUniqueLabels) {
            return Handlers_.OnUniqueLabels(req);
        }
        return MetabaseFailedResponse<TAsyncUniqueLabelsResponse>("not implemented");
    }

private:
    TStubMetabaseHandlers Handlers_;
};

/**
 * Stub cluster RPC implementation.
 */
class TStubMetabaseClusterRpc: public IMetabaseClusterRpc {
public:
    explicit TStubMetabaseClusterRpc(const THashMap<TString, TStubMetabaseHandlers>& nodes) {
        Addresses_.reserve(Nodes_.size());

        for (auto& [address, handlers]: nodes) {
            auto [it, inserted] = Nodes_.emplace(address, std::make_unique<TStubMetabaseRpc>(std::move(handlers)));
            Y_ENSURE(inserted, "duplicated address: " << address);
            Addresses_.push_back(it->first);
        }
    }

    IMetabaseRpc* Get(TStringBuf address) noexcept override {
        auto it = Nodes_.find(address);
        if (it == Nodes_.end()) {
            return nullptr;
        }
        return it->second.get();
    }

    IMetabaseRpc* GetAny() noexcept override {
        if (Nodes_.empty()) {
            return nullptr;
        }
        return Nodes_.begin()->second.get();
    }

    TString GetAnyAddress() const noexcept override {
        if (Nodes_.empty()) {
            return {};
        }
        return Nodes_.begin()->first;
    }

    const std::vector<TString>& Addresses() const noexcept override {
        return Addresses_;
    }

    void Add(TStringBuf) override {
        ythrow yexception() << "not implemented";
    }

    void Stop(bool) override {
    }

private:
    THashMap<TString, IMetabaseRpcPtr> Nodes_;
    TVector<TString> Addresses_;
};

} // namespace

// NOLINTNEXTLINE(performance-unnecessary-value-param): false positive
IMetabaseClusterRpcPtr StubMetabaseRpc(THashMap<TString, TStubMetabaseHandlers> nodes) {
    return std::make_shared<TStubMetabaseClusterRpc>(std::move(nodes));
}

} // namespace NSolomon::NDataProxy
