#pragma once

#include <solomon/services/dataproxy/lib/metabase/rpc.h>

namespace NSolomon::NDataProxy {

template <typename T>
T MetabaseFailedResponse(const TString& message) {
    using TType = typename NThreading::TFutureType<T>;
    return NThreading::MakeErrorFuture<TType>(std::make_exception_ptr(yexception() << message));
}

template <typename TResp, typename... Args>
TMetabaseAsyncResponse<TResp> MetabaseErrorResponse(Args... args) {
    return NThreading::MakeFuture(TErrorOr<std::unique_ptr<const TResp>, TMetabaseError>::FromError(
            TMetabaseError{std::forward<Args>(args)...}
    ));
}

template <typename TResp>
TMetabaseAsyncResponse<TResp> MetabaseResponse(const TResp& response) {
    return NThreading::MakeFuture(TErrorOr<std::unique_ptr<const TResp>, TMetabaseError>::FromValue(
            std::make_unique<const TResp>(response)));
}

/**
 * Metabase RPC handlers.
 *
 * If particular handler is not set, then method will return TFuture completed with an exception
 */
struct TStubMetabaseHandlers {
    std::function<TAsyncMetabaseStatusResponse(const yandex::solomon::metabase::TServerStatusRequest&)> OnServerStatus;
    std::function<TAsyncFindResponse(const yandex::solomon::metabase::FindRequest&)> OnFind;
    std::function<TAsyncCreateOneResponse(const yandex::solomon::metabase::CreateOneRequest&)> OnCreateOne;
    std::function<TAsyncCreateManyResponse(const yandex::solomon::metabase::CreateManyRequest&)> OnCreateMany;
    std::function<TAsyncResolveOneResponse(const yandex::solomon::metabase::ResolveOneRequest&)> OnResolveOne;
    std::function<TAsyncResolveManyResponse(const yandex::solomon::metabase::ResolveManyRequest&)> OnResolveMany;
    std::function<TAsyncDeleteManyResponse(const yandex::solomon::metabase::DeleteManyRequest&)> OnDeleteMany;
    std::function<TAsyncMetricNamesResponse(const yandex::solomon::metabase::MetricNamesRequest&)> OnMetricNames;
    std::function<TAsyncLabelValuesResponse(const yandex::solomon::metabase::TLabelValuesRequest&)> OnLabelValues;
    std::function<TAsyncLabelNamesResponse(const yandex::solomon::metabase::TLabelNamesRequest&)> OnLabelNames;
    std::function<TAsyncUniqueLabelsResponse(const yandex::solomon::metabase::TUniqueLabelsRequest&)> OnUniqueLabels;
};

/**
 * Create metabase cluster RPC.
 *
 * @param nodes  map of { node address -> handlers }
 * @return instance of fake metabase cluster RPC implementation.
 */
IMetabaseClusterRpcPtr StubMetabaseRpc(THashMap<TString, TStubMetabaseHandlers> nodes);

} // namespace NSolomon::NDataProxy
