#include <solomon/services/dataproxy/lib/metabase/cluster.h>
#include <solomon/services/dataproxy/lib/metabase/events.h>
#include <solomon/services/dataproxy/lib/metabase/stub/rpc_stub.h>
#include <solomon/services/dataproxy/lib/metabase/watcher.h>

#include <solomon/libs/cpp/actors/test_runtime/actor_runtime.h>

#include <library/cpp/actors/core/actor_bootstrapped.h>
#include <library/cpp/testing/gtest/gtest.h>

namespace {

using namespace NActors;
using namespace NSolomon::NDataProxy;
using namespace NSolomon;
using namespace yandex::solomon::metabase;

class TTestShardActor: public TActorBootstrapped<TTestShardActor> {
public:
    TTestShardActor(TAtomicCounter* counter)
        : Cnt{counter}
    {}

    void Bootstrap() {
        Become(&TThis::States);
    }

    STATEFN(States) {
        switch (ev->GetTypeRewrite()) {
            sFunc(TEvents::TEvPoison, OnPoison);
        }
    }

    void OnPoison() {
        Cnt->Inc();
        PassAway();
    }

    TAtomicCounter* Cnt;
};

class TTestShardActorFactory: public IShardActorFactory {
public:
    TTestShardActorFactory(TAtomicCounter* cnt)
        : Cnt_{cnt}
    {}

    std::unique_ptr<NActors::IActor> Create(
            TString,
            IMetabaseClusterRpcPtr,
            const TShardLocation&,
            size_t,
            NActors::TActorId) override {
        return std::make_unique<TTestShardActor>(Cnt_);
    }

private:
    TAtomicCounter* Cnt_;
};

class TMetabaseClusterTest: public ::testing::Test {
protected:
    void SetUp() override {
        ActorRuntime_ = TTestActorRuntime::CreateInited(1, false, true);
        EdgeId_ = ActorRuntime_->AllocateEdgeActor();

        Rpc_ = StubMetabaseRpc(
            {
                {"host-aaa", TStubMetabaseHandlers{
                    .OnServerStatus = [&](const auto&) {
                        return MetabaseResponse(TServerStatusResponse{});
                    },
                }},
            });

        auto* shardTtl = Config_.MutableShardTtl();
        shardTtl->SetUnit(yandex::solomon::config::SECONDS);
        shardTtl->SetValue(2);

        auto* cleanupInterval = Config_.MutableShardsCleanupInterval();
        cleanupInterval->SetUnit(yandex::solomon::config::SECONDS);
        cleanupInterval->SetValue(4);

        ShardActorFactory_ = std::make_unique<TTestShardActorFactory>(&PoisonCnt_);

        MetabaseCluster_ = ActorRuntime_->Register(MetabaseCluster(
                { "host-aaa" },
                Rpc_,
                std::move(ShardActorFactory_),
                Config_,
                {},
                {}).release());
        ActorRuntime_->WaitForBootstrap();
    }

    void TearDown() override {
        ActorRuntime_.Reset();
    }

protected:
    THolder<TTestActorRuntime> ActorRuntime_;
    TActorId EdgeId_;
    IMetabaseClusterRpcPtr Rpc_;
    TActorId MetabaseCluster_;
    TMetabaseConfig Config_;

    std::unique_ptr<IShardActorFactory> ShardActorFactory_;
    TAtomicCounter PoisonCnt_;
};

TEST_F(TMetabaseClusterTest, ShardsAreDeletedByTtl) {
    int expectedCnt = 0;
    ASSERT_EQ(PoisonCnt_.Val(), expectedCnt);

    std::vector<TShardInfoPtr> shards;
    auto addShard = [&](auto&& shardId, auto&& service) {
        TShardKey key{"prj", TShardSubKey{"ctr", service}};
        auto shard = std::make_shared<TShardInfo>(shardId, "host-aaa", std::move(key), true);

        shards.push_back(std::move(shard));
    };

    addShard(1, "service1");
    addShard(2, "service2");
    addShard(3, "service3");

    auto addShardsEv = std::make_unique<TMetabaseWatcherEvents::TStateChanged>();
    addShardsEv->Updated = std::move(shards);

    ActorRuntime_->Send(MetabaseCluster_, THolder(addShardsEv.release()));

    for (size_t i = 1; i <= 3; ++i) {
        auto findReq = std::make_unique<TMetabaseEvents::TFindReq>();
        findReq->ShardSelector = TShardSelector(
                "prj",
                {"cluster", "ctr", false},
                {"service", "service" + ToString(i), false});
        findReq->Message = std::make_shared<typename TMetabaseEvents::TFindReq::TProtoMsg>();

        ActorRuntime_->Send(MetabaseCluster_, THolder(findReq.release()));

        ASSERT_EQ(PoisonCnt_.Val(), expectedCnt);
        ActorRuntime_->AdvanceCurrentTime(TDuration::Seconds(3));
        ActorRuntime_->DispatchEvents({}, TDuration::Seconds(1));

        ASSERT_EQ(PoisonCnt_.Val(), ++expectedCnt);
    }
}

} // namespace
