#include "rpc_stub.h"

#include <solomon/services/dataproxy/lib/message_cache/cache_actor.h>
#include <solomon/services/dataproxy/lib/metabase/events.h>
#include <solomon/services/dataproxy/lib/metabase/requester.h>
#include <solomon/services/dataproxy/lib/metabase/shard_actor.h>
#include <solomon/services/dataproxy/lib/metabase/stub/rpc_stub.h>

#include <solomon/libs/cpp/actors/scheduler/scheduler.h>
#include <solomon/libs/cpp/actors/test_runtime/actor_runtime.h>

#include <library/cpp/actors/core/event.h>
#include <library/cpp/testing/gtest/gtest.h>
#include <library/cpp/testing/gtest_extensions/matchers.h>

#include <google/protobuf/util/message_differencer.h>

using namespace NSolomon;
using namespace NDataProxy;

using google::protobuf::util::MessageDifferencer;
using namespace testing;
using namespace yandex::solomon::metabase;
using namespace yandex::solomon::model;

class TRequesterTest: public ::testing::Test {
public:
    TRequesterTest() {

    }

protected:
    void SetUp() override {
        ActorRuntime_ = TTestActorRuntime::CreateInited(1, false, true);
        ActorRuntime_->SetLogPriority(ELogComponent::TraceEvents, NActors::NLog::PRI_TRACE);
        ActorRuntime_->WaitForBootstrap();

        EdgeId_ = ActorRuntime_->AllocateEdgeActor();
        RpcStub_ = {};

        SchedulerId_ = ActorRuntime_->Register(
                CreateScheduler(ActorRuntime_->GetCurrentTime(), TDuration::Seconds(1)).release()
        );
        ActorRuntime_->WaitForBootstrap();

        CacheId_ = ActorRuntime_->Register(CreateMessageCacheStub().release());
        ShardActorFactory_ = CreateShardActorFactory("", SchedulerId_);
    }

    void TearDown() override {
        if (!ShardActors_.empty()) {
            for (auto s: ShardActors_) {
                ActorRuntime_->Send(s, EdgeId_, MakeHolder<NActors::TEvents::TEvPoison>());
                auto ev = ActorRuntime_->GrabEdgeEvent<NActors::TEvents::TEvPoisonTaken>(EdgeId_, TDuration::Seconds(5));
            }
        }

        ActorRuntime_.Reset();
    }

    template <typename TEvent>
    NActors::TActorId SendRequest(
            ui32 cookie,
            const TVector<TShardLocation>& shards)
    {
        TVector<TShardActorId> shardActors(::Reserve(shards.size()));
        for (const auto& s: shards) {
            auto shardActorId = ActorRuntime_->Register(
                ShardActorFactory_->Create("someProject", RpcStub_.Rpc, s, 2000, CacheId_).release()
            );
            ShardActors_.emplace_back(shardActorId);
            shardActors.emplace_back(TShardActorId{s.Id, shardActorId});

            ActorRuntime_->WaitForBootstrap();
        }

        auto requesterId = ActorRuntime_->Register(ShardsRequester<TEvent>(std::move(shardActors), SchedulerId_).release());


        auto ev = std::make_unique<TEvent>();
        ev->Deadline = ActorRuntime_->GetCurrentTime() + TDuration::Seconds(30);

        auto msg = std::make_shared<typename TEvent::TProtoMsg>();
        msg->mutable_sliceoptions()->set_limit(90'000);
        ev->Message = std::move(msg);

        ActorRuntime_->Send(new NActors::IEventHandle(requesterId, EdgeId_, ev.release(), 0, cookie));

        return requesterId;
    }

    template <typename TEvent>
    typename TEvent::TPtr ReceiveResponse() {
        return ActorRuntime_->GrabEdgeEvent<TEvent>(EdgeId_, TDuration::Seconds(30));
    }

protected:
    THolder<TTestActorRuntime> ActorRuntime_;
    NMonitoring::TMetricRegistry Registry_;
    NActors::TActorId EdgeId_;
    NActors::TActorId SchedulerId_;
    NActors::TActorId CacheId_;
    TRpcStub RpcStub_;

    TVector<NActors::TActorId> ShardActors_;
    std::shared_ptr<IShardActorFactory> ShardActorFactory_;
};

TEST_F(TRequesterTest, ShardsRequester_Ok) {
    TShardLocation shardA{123, "find-ok"};
    TShardLocation shardB{321, "find-ok-2"};
    const ui32 cookie = 777;

    auto requesterId = SendRequest<TMetabaseEvents::TFindReq>(cookie, {shardA, shardB});

    // shardA
    auto findRespA = ReceiveResponse<TMetabaseEvents::TFindResp>();
    ASSERT_TRUE(findRespA);
    EXPECT_EQ(findRespA->Cookie, cookie);
    EXPECT_TRUE(MessageDifferencer::Equals(RpcStub_.OkResponse, *(findRespA->Get()->Message)));

    // shardB
    auto findRespB = ReceiveResponse<TMetabaseEvents::TFindResp>();
    ASSERT_TRUE(findRespB);
    EXPECT_EQ(findRespB->Cookie, cookie);
    EXPECT_TRUE(MessageDifferencer::Equals(RpcStub_.OkResponse, *(findRespB->Get()->Message)));

    auto doneResp = ReceiveResponse<TMetabaseEvents::TDone>();
    ASSERT_TRUE(doneResp);

    // requester must die after sending last response
    EXPECT_TRUE(ActorRuntime_->FindActor(requesterId) == nullptr);
}

TEST_F(TRequesterTest, ShardsRequester_NotFound) {
    TShardLocation shardA{123, "find-ok"};
    TShardLocation shardB{321, "find-not-found"};
    const ui32 cookie = 777;

    auto requesterId = SendRequest<TMetabaseEvents::TFindReq>(cookie, {shardA, shardB});

    // shardA
    auto findResp = ReceiveResponse<TMetabaseEvents::TFindResp>();
    ASSERT_TRUE(findResp);
    EXPECT_EQ(findResp->Cookie, cookie);
    EXPECT_TRUE(MessageDifferencer::Equals(RpcStub_.OkResponse, *(findResp->Get()->Message)));

    // shardB
    auto errorResp = ReceiveResponse<TMetabaseEvents::TError>();
    ASSERT_TRUE(errorResp);
    EXPECT_EQ(errorResp->Cookie, cookie);
    EXPECT_EQ(errorResp->Get()->RpcCode, grpc::StatusCode::OK);
    EXPECT_EQ(errorResp->Get()->MetabaseCode, RpcStub_.NotFoundResponse.status());
    EXPECT_EQ(errorResp->Get()->Message, RpcStub_.NotFoundResponse.statusmessage());

    auto doneResp = ReceiveResponse<TMetabaseEvents::TDone>();
    ASSERT_TRUE(doneResp);

    // requester must die after sending last response
    EXPECT_TRUE(ActorRuntime_->FindActor(requesterId) == nullptr);
}

TEST_F(TRequesterTest, ShardsRequester_RpcError) {
    TShardLocation shardA{123, "find-ok"};
    TShardLocation shardB{321, "find-rpc-error"};
    const ui32 cookie = 777;

    auto requesterId = SendRequest<TMetabaseEvents::TFindReq>(cookie, {shardA, shardB});

    // shardA
    auto findResp = ReceiveResponse<TMetabaseEvents::TFindResp>();
    ASSERT_TRUE(findResp);
    EXPECT_EQ(findResp->Cookie, cookie);
    EXPECT_TRUE(MessageDifferencer::Equals(RpcStub_.OkResponse, *(findResp->Get()->Message)));

    // shardB
    auto errorResp = ReceiveResponse<TMetabaseEvents::TError>();
    ASSERT_TRUE(errorResp);
    EXPECT_EQ(errorResp->Cookie, cookie);
    EXPECT_EQ(errorResp->Get()->RpcCode, grpc::StatusCode::INTERNAL);
    EXPECT_EQ(errorResp->Get()->MetabaseCode, EMetabaseStatusCode::INTERNAL_ERROR);
    EXPECT_EQ(errorResp->Get()->Message, "internal error");

    auto doneResp = ReceiveResponse<TMetabaseEvents::TDone>();
    ASSERT_TRUE(doneResp);

    // requester must die after sending last response
    EXPECT_TRUE(ActorRuntime_->FindActor(requesterId) == nullptr);
}

TEST_F(TRequesterTest, ShardsRequester_LegitimateTimeout) {
    TShardLocation shardA{321, "find-timeout"};
    const ui32 cookie = 777;

    auto requesterId = SendRequest<TMetabaseEvents::TFindReq>(cookie, {shardA});

    // shardA
    auto errorResp = ReceiveResponse<TMetabaseEvents::TError>();
    ASSERT_TRUE(errorResp);
    EXPECT_EQ(errorResp->Cookie, cookie);
    EXPECT_EQ(errorResp->Get()->RpcCode, grpc::StatusCode::DEADLINE_EXCEEDED);
    EXPECT_EQ(errorResp->Get()->MetabaseCode, EMetabaseStatusCode::DEADLINE_EXCEEDED);
    EXPECT_THAT(errorResp->Get()->Message, StartsWith("request timed out at "));

    auto doneResp = ReceiveResponse<TMetabaseEvents::TDone>();
    ASSERT_TRUE(doneResp);

    // requester must die after sending last response
    EXPECT_TRUE(ActorRuntime_->FindActor(requesterId) == nullptr);
}

TEST_F(TRequesterTest, ShardsRequester_Fail) {
    TShardLocation shardA{123, "find-ok"};
    TShardLocation shardB{321, "find-fail"};
    const ui32 cookie = 777;

    auto requesterId = SendRequest<TMetabaseEvents::TFindReq>(cookie, {shardA, shardB});

    // shardA
    auto findResp = ReceiveResponse<TMetabaseEvents::TFindResp>();
    ASSERT_TRUE(findResp);
    EXPECT_EQ(findResp->Cookie, cookie);
    EXPECT_TRUE(MessageDifferencer::Equals(RpcStub_.OkResponse, *(findResp->Get()->Message)));

    // shardB
    auto errorResp = ReceiveResponse<TMetabaseEvents::TError>();
    ASSERT_TRUE(errorResp);
    EXPECT_EQ(errorResp->Cookie, cookie);
    EXPECT_EQ(errorResp->Get()->RpcCode, grpc::StatusCode::UNKNOWN);
    EXPECT_EQ(errorResp->Get()->MetabaseCode, EMetabaseStatusCode::INTERNAL_ERROR);
    EXPECT_TRUE(errorResp->Get()->Message.EndsWith("unexpected error Y"));

    auto doneResp = ReceiveResponse<TMetabaseEvents::TDone>();
    ASSERT_TRUE(doneResp);

    // requester must die after sending last response
    EXPECT_TRUE(ActorRuntime_->FindActor(requesterId) == nullptr);
}

TEST_F(TRequesterTest, ShardsRequester_Retryable) {
    TShardLocation shardA{123, "find-ok"};
    TShardLocation shardB{321, "find-retryable-error"};
    const ui32 cookie = 777;

    auto requesterId = SendRequest<TMetabaseEvents::TFindReq>(cookie, {shardA, shardB});

    // shardA
    auto findRespA = ReceiveResponse<TMetabaseEvents::TFindResp>();
    ASSERT_TRUE(findRespA);
    EXPECT_EQ(findRespA->Cookie, cookie);
    EXPECT_TRUE(MessageDifferencer::Equals(RpcStub_.OkResponse, *(findRespA->Get()->Message)));

    // shardB
    auto findRespB = ReceiveResponse<TMetabaseEvents::TFindResp>();
    ASSERT_TRUE(findRespB);
    EXPECT_EQ(findRespB->Cookie, cookie);
    EXPECT_TRUE(MessageDifferencer::Equals(RpcStub_.OkResponse, *(findRespB->Get()->Message)));

    auto doneResp = ReceiveResponse<TMetabaseEvents::TDone>();
    ASSERT_TRUE(doneResp);

    // requester must die after sending last response
    EXPECT_TRUE(ActorRuntime_->FindActor(requesterId) == nullptr);
}
