#include "rpc_stub.h"

#include <solomon/services/dataproxy/config/cache_config.pb.h>

#include <solomon/services/dataproxy/lib/message_cache/cache_actor.h>
#include <solomon/services/dataproxy/lib/metabase/events.h>
#include <solomon/services/dataproxy/lib/metabase/shard_actor.h>
#include <solomon/services/dataproxy/lib/metabase/stub/rpc_stub.h>

#include <solomon/libs/cpp/actors/scheduler/scheduler.h>
#include <solomon/libs/cpp/actors/test_runtime/actor_runtime.h>

#include <library/cpp/testing/gtest/gtest.h>

#include <google/protobuf/util/message_differencer.h>

#include <array>

using namespace NActors;
using namespace NSolomon::NDataProxy;
using namespace NSolomon;
using namespace yandex::solomon::metabase;
using google::protobuf::util::MessageDifferencer;

TCacheConfig ConstructCacheConfig() {
    TCacheConfig cacheConfig;

    auto* prjLimit = cacheConfig.MutableProjectLimit();
    prjLimit->SetValue(10);
    prjLimit->SetUnit(yandex::solomon::config::MEGABYTES);

    auto* totalLimit = cacheConfig.MutableTotalLimit();
    totalLimit->SetValue(100);
    prjLimit->SetUnit(yandex::solomon::config::MEGABYTES);

    auto* exp = cacheConfig.MutableExpireAfter();
    exp->SetValue(10);
    exp->SetUnit(yandex::solomon::config::TimeUnit::MINUTES);

    auto* refresh = cacheConfig.MutableRefreshInterval();
    refresh->SetValue(10);
    refresh->SetUnit(yandex::solomon::config::TimeUnit::MINUTES);

    return cacheConfig;
}

class TMetabaseShardActorTest: public ::testing::Test {
protected:
    void SetUp() override {
        Metrics_ = std::make_shared<NMonitoring::TMetricRegistry>();

        ActorRuntime_ = TTestActorRuntime::CreateInited(1, false, true);
        ActorRuntime_->WaitForBootstrap();

        EdgeId_ = ActorRuntime_->AllocateEdgeActor();

        SchedulerId_ = ActorRuntime_->Register(
                CreateScheduler(ActorRuntime_->GetCurrentTime(), TDuration::Seconds(1)));
        ActorRuntime_->WaitForBootstrap();

        CacheId_ = ActorRuntime_->Register(
                CreateMessageCache(ConstructCacheConfig(), Metrics_, ELogComponent::MetaCache, ""));
        ActorRuntime_->WaitForBootstrap();

        ShardActorFactory_ = CreateShardActorFactory("", SchedulerId_);
        RpcStub_ = {};
    }

    void TearDown() override {
        ActorRuntime_.Reset();
    }

    void DispatchEvents() {
        // the time inside DispatchEvents (and in the whole testing runtime, for that matter) is artificial
        ActorRuntime_->DispatchEvents({}, TDuration::Seconds(30));
    };

    template <typename TEvent>
    NActors::TActorId SendRequest(ui32 cookie, const TShardLocation& shard) {
        auto shardActor = ActorRuntime_->Register(
            ShardActorFactory_->Create("someProject", RpcStub_.Rpc, shard, 2000, CacheId_).release()
        );
        ActorRuntime_->WaitForBootstrap();

        auto ev = std::make_unique<TEvent>();
        ev->Deadline = TDuration::Seconds(30).ToDeadLine();
        auto msg = std::make_shared<typename TEvent::TProtoMsg>();
        msg->set_shard_id(shard.Id);
        msg->mutable_sliceoptions()->set_limit(90'000);
        ev->Message = std::move(msg);
        // ev->Message = ...; actually not needed for these tests
        ActorRuntime_->Send(new NActors::IEventHandle(shardActor, EdgeId_, ev.release(), 0, cookie));

        return shardActor;
    }

    template <typename TEvent>
    typename TEvent::TPtr ReceiveResponse() {
        return ActorRuntime_->GrabEdgeEvent<TEvent>(EdgeId_, TDuration::Seconds(30));
    }

protected:
    THolder<TTestActorRuntime> ActorRuntime_;
    TActorId EdgeId_;
    TActorId SchedulerId_;
    TActorId CacheId_;
    std::shared_ptr<NMonitoring::TMetricRegistry> Metrics_;
    std::shared_ptr<IShardActorFactory> ShardActorFactory_;
    TRpcStub RpcStub_;
};

TEST_F(TMetabaseShardActorTest, LocationIsUpdated) {
    const TShardLocation shard{1ul, "host-aaa"};
    std::vector<TString> locations;
    FindResponse OkResponse_;
    OkResponse_.set_status(EMetabaseStatusCode::OK);

    auto rpc = StubMetabaseRpc({
        {"host-aaa", TStubMetabaseHandlers{
            .OnFind = [&](const auto&) {
                locations.emplace_back("host-aaa");

                return MetabaseResponse(OkResponse_);
            },
        }},
        {"host-bbb", TStubMetabaseHandlers{
            .OnFind = [&](const auto&) {
                locations.emplace_back("host-bbb");

                return MetabaseResponse(OkResponse_);
            },
        }},
    });

    auto shardActorId = ActorRuntime_->Register(
            ShardActorFactory_->Create("projId", rpc, shard, 2000, CacheId_).release());
    ActorRuntime_->WaitForBootstrap();

    {
        auto ev = std::make_unique<TMetabaseEvents::TFindReq>();
        // to distinguish requests inside ShardActor
        auto msg = std::make_shared<typename TMetabaseEvents::TFindReq::TProtoMsg>();
        msg->set_fillmetricname(false);
        ev->Message = std::move(msg);
        ev->Deadline = TDuration::Seconds(30).ToDeadLine();

        ActorRuntime_->Send(new IEventHandle(shardActorId, EdgeId_, ev.release(), 0, 777));
    }

    auto ev1 = ActorRuntime_->GrabEdgeEvent<TMetabaseEvents::TFindResp>(EdgeId_, TDuration::Seconds(10));
    ASSERT_TRUE(ev1);

    TShardKey key;
    ActorRuntime_->Send(shardActorId, MakeHolder<TMetabaseShardActorEvents::TShardUpdate>(
        std::make_shared<TShardInfo>(1ul, "host-bbb", std::move(key), true)));

    {
        auto ev = std::make_unique<TMetabaseEvents::TFindReq>();
        auto msg = std::make_shared<typename TMetabaseEvents::TFindReq::TProtoMsg>();
        msg->set_fillmetricname(true);
        ev->Message = std::move(msg);
        ev->Deadline = TDuration::Seconds(30).ToDeadLine();

        ActorRuntime_->Send(new IEventHandle(shardActorId, EdgeId_, ev.release(), 0, 777));
    }

    auto ev2 = ActorRuntime_->GrabEdgeEvent<TMetabaseEvents::TFindResp>(EdgeId_, TDuration::Seconds(10));
    ASSERT_TRUE(ev2);

    ASSERT_EQ(locations.size(), 2ull);
    EXPECT_EQ(locations[0], "host-aaa");
    EXPECT_EQ(locations[1], "host-bbb");
}

TEST_F(TMetabaseShardActorTest, CacheIsUsed) {
    const TShardLocation shard{1ul, "host-aaa"};
    size_t requestsToRpc{0};
    FindResponse OkResponse_;
    OkResponse_.set_status(EMetabaseStatusCode::OK);

    auto rpc = StubMetabaseRpc(
        {
            {"host-aaa", TStubMetabaseHandlers{
                .OnFind = [&](const auto&) {
                    ++requestsToRpc;

                    return MetabaseResponse(OkResponse_);
                },
            }},
        });

    auto shardActorId = ActorRuntime_->Register(
            ShardActorFactory_->Create("projId", rpc, shard, 2000, CacheId_).release());
    ActorRuntime_->WaitForBootstrap();

    {
        auto ev = std::make_unique<TMetabaseEvents::TFindReq>();
        // proto msgs in this test should have the same values in order to have the same hash
        auto msg = std::make_shared<typename TMetabaseEvents::TFindReq::TProtoMsg>();
        msg->set_fillmetricname(true);
        ev->Message = std::move(msg);
        ev->Deadline = TDuration::Seconds(30).ToDeadLine();

        ActorRuntime_->Send(new IEventHandle(shardActorId, EdgeId_, ev.release(), 0, 777));
    }

    auto ev1 = ActorRuntime_->GrabEdgeEvent<TMetabaseEvents::TFindResp>(EdgeId_, TDuration::Seconds(10));
    ASSERT_TRUE(ev1);

    {
        auto ev = std::make_unique<TMetabaseEvents::TFindReq>();
        auto msg = std::make_shared<typename TMetabaseEvents::TFindReq::TProtoMsg>();
        msg->set_fillmetricname(true);
        ev->Message = std::move(msg);
        ev->Deadline = TDuration::Seconds(30).ToDeadLine();

        ActorRuntime_->Send(new IEventHandle(shardActorId, EdgeId_, ev.release(), 0, 777));
    }

    auto ev2 = ActorRuntime_->GrabEdgeEvent<TMetabaseEvents::TFindResp>(EdgeId_, TDuration::Seconds(10));
    ASSERT_TRUE(ev2);

    ASSERT_EQ(requestsToRpc, 1ull);
    ASSERT_EQ(ev1->Get()->Message->status(), EMetabaseStatusCode::OK);
    ASSERT_TRUE(MessageDifferencer::Equals(*(ev1->Get()->Message), *(ev2->Get()->Message)));

    // wait for the actor destroying its internal data structures
    ActorRuntime_->Send(new IEventHandle(shardActorId, EdgeId_, new TEvents::TEvPoison));
    auto poisonTaken = ActorRuntime_->GrabEdgeEvent<TEvents::TEvPoisonTaken>(EdgeId_, TDuration::Seconds(20));
    ASSERT_TRUE(poisonTaken);
}

TEST_F(TMetabaseShardActorTest, Subscriptions) {
    const TShardLocation shard{1ul, "host-aaa"};
    size_t requestsToRpc{0};

    std::array<NActors::TActorId, 3> subscribers;
    for (size_t i = 0; i != subscribers.size(); ++i) {
        subscribers[i] = ActorRuntime_->AllocateEdgeActor();
    }

    using TErrOrValue = TMetabaseAsyncResponse<FindResponse>::value_type;
    auto promise = NThreading::NewPromise<TErrOrValue>();

    auto rpc = StubMetabaseRpc(
            {
                    {"host-aaa", TStubMetabaseHandlers{
                            .OnFind = [&](const auto&) {
                                ++requestsToRpc;
                                return promise.GetFuture();
                            },
                    }},
            });

    auto shardActorId = ActorRuntime_->Register(
            ShardActorFactory_->Create("projId", rpc, shard, 2000, CacheId_).release());
    ActorRuntime_->WaitForBootstrap();

    {
        auto createMsg = []() {
            auto ev = std::make_unique<TMetabaseEvents::TFindReq>();
            // proto msgs in this test should have the same values in order to have the same hash
            auto msg = std::make_shared<typename TMetabaseEvents::TFindReq::TProtoMsg>();
            msg->set_fillmetricname(true);
            ev->Message = std::move(msg);
            ev->Deadline = TDuration::Seconds(30).ToDeadLine();

            return ev;
        };

        ActorRuntime_->Send(new IEventHandle(shardActorId, EdgeId_, createMsg().release(), 0, 777));

        for (auto subId: subscribers) {
            ActorRuntime_->Send(new IEventHandle(shardActorId, subId, createMsg().release(), 0, 777));
        }
    }

    auto okResponse = std::make_unique<FindResponse>();
    okResponse->set_status(EMetabaseStatusCode::OK);
    promise.SetValue(TErrOrValue::FromValue(std::move(okResponse)));

    auto ev = ActorRuntime_->GrabEdgeEvent<TMetabaseEvents::TFindResp>(EdgeId_, TDuration::Seconds(10));
    ASSERT_TRUE(ev);

    std::array<TMetabaseEvents::TFindResp::TPtr, subscribers.size()> subResps;
    for (size_t i = 0; i != subResps.size(); ++i) {
        subResps[i] = ActorRuntime_->GrabEdgeEvent<TMetabaseEvents::TFindResp>(subscribers[i], TDuration::Seconds(10));
        ASSERT_TRUE(subResps[i]);
    }

    ASSERT_EQ(requestsToRpc, 1ull);
    ASSERT_EQ(ev->Get()->Message->status(), EMetabaseStatusCode::OK);
    for (const auto& resp: subResps) {
        ASSERT_TRUE(MessageDifferencer::Equals(*(ev->Get()->Message), *(resp->Get()->Message)));
    }
}

TEST_F(TMetabaseShardActorTest, InflightLimitAndPostponedRequests) {
    CacheId_ = ActorRuntime_->Register(CreateMessageCacheStub().release());
    const size_t inflightLimit = 100;

    using TErrOrValue = TMetabaseAsyncResponse<FindResponse>::value_type;
    TVector<NThreading::TPromise<TErrOrValue>> promises;

    auto rpc = StubMetabaseRpc(
            {
                    {"host-aaa", TStubMetabaseHandlers{
                            .OnFind = [&](const auto&) mutable {
                                promises.push_back(NThreading::NewPromise<TErrOrValue>());
                                return promises.back().GetFuture();
                            },
                    }},
            });

    const TShardLocation shard{1ul, "host-aaa"};
    auto shardActorId = ActorRuntime_->Register(
            ShardActorFactory_->Create("projId", rpc, shard, inflightLimit, CacheId_).release());
    ActorRuntime_->WaitForBootstrap();

    ui32 msgIdx = 0;
    auto createMsg = [&]() {
        auto ev = std::make_unique<TMetabaseEvents::TFindReq>();
        auto msg = std::make_shared<typename TMetabaseEvents::TFindReq::TProtoMsg>();
        msg->set_fillmetricname(true);
        msg->mutable_sliceoptions()->set_limit(++msgIdx);

        ev->Message = std::move(msg);
        ev->Deadline = TDuration::Seconds(30).ToDeadLine();

        return ev;
    };

    for (size_t i = 0; i != inflightLimit + inflightLimit / 2; ++i) {
        ActorRuntime_->Send(new IEventHandle(shardActorId, EdgeId_, createMsg().release(), 0, 777));
    }

    DispatchEvents();
    ASSERT_EQ(promises.size(), inflightLimit);

    auto createResp = []() {
        auto okResponse = std::make_unique<FindResponse>();
        okResponse->set_status(EMetabaseStatusCode::OK);

        return okResponse;
    };

    for (size_t i = 0; i != inflightLimit; ++i) {
        promises.back().SetValue(TErrOrValue::FromValue(createResp()));
        promises.pop_back();
    }
    ASSERT_EQ(promises.size(), 0ull);

    // check that postponed requests are processed
    DispatchEvents();
    ASSERT_EQ(promises.size(), inflightLimit / 2);

    // , but they are in the inflight state. So let's check that we don't go beyond the limit
    for (size_t i = 0; i != inflightLimit; ++i) {
        ActorRuntime_->Send(new IEventHandle(shardActorId, EdgeId_, createMsg().release(), 0, 777));
    }

    DispatchEvents();
    ASSERT_EQ(promises.size(), inflightLimit);

    for (size_t i = 0; i != inflightLimit; ++i) {
        promises.back().SetValue(TErrOrValue::FromValue(createResp()));
        promises.pop_back();
    }
    ASSERT_EQ(promises.size(), 0ull);

    // check that postponed requests are processed
    DispatchEvents();
    ASSERT_EQ(promises.size(), inflightLimit / 2);

    for (size_t i = 0; i != inflightLimit / 2; ++i) {
        promises.back().SetValue(TErrOrValue::FromValue(createResp()));
        promises.pop_back();
    }
    ASSERT_EQ(promises.size(), 0ull);

    DispatchEvents();
    ASSERT_EQ(promises.size(), 0ull);
}

TEST_F(TMetabaseShardActorTest, ShardActor_Ok) {
    TShardLocation shard{123, "find-ok"};
    const ui32 cookie = 777;

    SendRequest<TMetabaseEvents::TFindReq>(cookie, shard);

    auto findResp = ReceiveResponse<TMetabaseEvents::TFindResp>();
    ASSERT_TRUE(findResp);
    EXPECT_EQ(findResp->Cookie, cookie);
    EXPECT_TRUE(MessageDifferencer::Equals(RpcStub_.OkResponse, *(findResp->Get()->Message)));
}

TEST_F(TMetabaseShardActorTest, ShardActor_NotFound) {
    TShardLocation shard{123, "find-not-found"};
    const ui32 cookie = 777;

    SendRequest<TMetabaseEvents::TFindReq>(cookie, shard);

    auto errorResp = ReceiveResponse<TMetabaseEvents::TError>();
    ASSERT_TRUE(errorResp);
    EXPECT_EQ(errorResp->Cookie, cookie);
    EXPECT_EQ(errorResp->Get()->RpcCode, grpc::StatusCode::OK);
    EXPECT_EQ(errorResp->Get()->MetabaseCode, RpcStub_.NotFoundResponse.status());
    EXPECT_EQ(errorResp->Get()->Message, RpcStub_.NotFoundResponse.statusmessage());
}

TEST_F(TMetabaseShardActorTest, ShardActor_RpcError) {
    TShardLocation shard{123, "find-rpc-error"};
    const ui32 cookie = 777;

    SendRequest<TMetabaseEvents::TFindReq>(cookie, shard);

    auto errorResp = ReceiveResponse<TMetabaseEvents::TError>();
    ASSERT_TRUE(errorResp);
    EXPECT_EQ(errorResp->Cookie, cookie);
    EXPECT_EQ(errorResp->Get()->RpcCode, grpc::StatusCode::INTERNAL);
    EXPECT_EQ(errorResp->Get()->MetabaseCode, EMetabaseStatusCode::INTERNAL_ERROR);
    EXPECT_EQ(errorResp->Get()->Message, "internal error");
}

TEST_F(TMetabaseShardActorTest, ShardActor_Fail) {
    TShardLocation shard{123, "find-fail"};
    const ui32 cookie = 777;

    SendRequest<TMetabaseEvents::TFindReq>(cookie, shard);

    auto errorResp = ReceiveResponse<TMetabaseEvents::TError>();
    ASSERT_TRUE(errorResp);
    EXPECT_EQ(errorResp->Cookie, cookie);
    EXPECT_EQ(errorResp->Get()->RpcCode, grpc::StatusCode::UNKNOWN);
    EXPECT_EQ(errorResp->Get()->MetabaseCode, EMetabaseStatusCode::INTERNAL_ERROR);
    EXPECT_TRUE(errorResp->Get()->Message.EndsWith("unexpected error Y"));
}

TEST_F(TMetabaseShardActorTest, ShardActor_Retryable) {
    TShardLocation shard{123, "find-retryable-error"};
    const ui32 cookie = 777;

    SendRequest<TMetabaseEvents::TFindReq>(cookie, shard);

    auto findResp = ReceiveResponse<TMetabaseEvents::TFindResp>();
    ASSERT_TRUE(findResp);
    EXPECT_EQ(findResp->Cookie, cookie);
    EXPECT_TRUE(MessageDifferencer::Equals(RpcStub_.OkResponse, *(findResp->Get()->Message)));
}

TEST_F(TMetabaseShardActorTest, ShardActor_Paging) {
    TShardLocation shard{123, "find-paging"};
    const ui32 cookie = 777;


    FindResponse mergedResponse;
    mergedResponse.CopyFrom(RpcStub_.OkResponse);
    mergedResponse.set_totalcount(100);
    for (int k = 0; k < 19; ++k) {  // metrics_size == 5, total_count == 100, add 95 metrics
        for (int i = 0; i < 5; ++i) {
            auto* m = mergedResponse.add_metrics();
            m->set_type(static_cast<MetricType>(static_cast<int>(MetricType::DGAUGE) + i));
            m->set_name(TStringBuilder{} << "metric" << i);

            for (int j = 0; j < 3; ++j) {
                auto* l = m->add_labels();
                l->set_key(TStringBuilder{} << "key" << j);
                l->set_value(TStringBuilder{} << "value" << j);
            }
        }
    }

    SendRequest<TMetabaseEvents::TFindReq>(cookie, shard);

    auto findResp = ReceiveResponse<TMetabaseEvents::TFindResp>();
    ASSERT_TRUE(findResp);
    EXPECT_EQ(findResp->Cookie, cookie);
    EXPECT_TRUE(MessageDifferencer::Equals(mergedResponse, *(findResp->Get()->Message)));
}
