#include <solomon/services/dataproxy/lib/metabase/watcher.h>
#include <solomon/services/dataproxy/lib/metabase/stub/rpc_stub.h>

#include <solomon/libs/cpp/actors/test_runtime/actor_runtime.h>

#include <library/cpp/testing/gtest/gtest.h>

using namespace NSolomon;
using namespace NDataProxy;

using namespace yandex::solomon::metabase;
using namespace yandex::solomon::model;

namespace {

class TFakeMetabaseNode {
public:
    explicit TFakeMetabaseNode(TString address)
        : Address_{std::move(address)}
    {
    }

    TShardInfoPtr AddShard(TShardId id, TString project, TString cluster, TString service) {
        TShardKey key{std::move(project), {std::move(cluster), std::move(service)}};
        auto s = std::make_shared<TShardInfo>(id, Address_, std::move(key), true);

        Shards_[id] = s;

        return s;
    }

    TShardInfoPtr RemoveShard(TShardId shardId) {
        auto node = Shards_.extract(shardId);
        return node.mapped();
    }

    TServerStatusResponse Status() const {
        TServerStatusResponse resp;
        resp.SetStatus(EMetabaseStatusCode::OK);

        ui64 hash = 0;
        for (auto& [shardId, shard]: Shards_) {
            hash = CombineHashes<ui64>(hash, shardId);

            TShardStatus* shardProto = resp.AddPartitionStatus();
            shardProto->SetNumId(shardId);
            shardProto->SetReady(shard->Ready);

            auto addLabel = [shardProto](const TString& key, const TString& value) {
                auto label = shardProto->AddLabels();
                label->set_key(key);
                label->set_value(value);
            };

            addLabel("project", shard->Key.Project);
            addLabel("cluster", shard->Key.SubKey.Cluster);
            addLabel("service", shard->Key.SubKey.Service);
        }

        resp.SetShardIdsHash(hash);
        return resp;
    }

    const TString& Address() const noexcept {
        return Address_;
    }

private:
    TString Address_;
    std::map<TShardId, TShardInfoPtr> Shards_;
};

IMetabaseClusterRpcPtr MakeFakeRpc(const std::vector<TFakeMetabaseNode*>& nodes) {
    THashMap<TString, TStubMetabaseHandlers> handlers;
    for (TFakeMetabaseNode* node: nodes) {
        handlers[node->Address()] = TStubMetabaseHandlers{
                .OnServerStatus = [node](const TServerStatusRequest& req) {
                    Y_ENSURE(req.GetDeadlineMillis() != 0, "deadline is not set");
                    return MetabaseResponse(node->Status());
                }
        };
    }
    return StubMetabaseRpc(std::move(handlers));
}

auto FindShard(const std::vector<TShardInfoPtr>& shards, TShardId id) {
    return std::find_if(shards.begin(), shards.end(), [id](const TShardInfoPtr& s) {
        return s->Id == id;
    });
}

auto FindShard(const std::vector<TShardLocation>& locations, TShardId id) {
    return std::find_if(locations.begin(), locations.end(), [id](const TShardLocation& l) {
        return l.Id == id;
    });
};


} // namespace

class TMetabaseWatcherTest: public ::testing::Test {
protected:
    void SetUp() override {
        ActorRuntime_ = TTestActorRuntime::CreateInited(1, false, true);
        ActorRuntime_->WaitForBootstrap();
        EdgeId_ = ActorRuntime_->AllocateEdgeActor();
    }

    void TearDown() override {
        ActorRuntime_.Reset();
    }

    template <typename TEvent>
    typename TEvent::TPtr ReceiveResponse() {
        return ActorRuntime_->GrabEdgeEvent<TEvent>(EdgeId_, TDuration::Seconds(10));
    }

protected:
    THolder<TTestActorRuntime> ActorRuntime_;
    NActors::TActorId EdgeId_;
};

TEST_F(TMetabaseWatcherTest, GetChangesOverSubscription) {
    TFakeMetabaseNode nodeA{"node-a"};
    auto a1 = nodeA.AddShard(1, "solomon", "production", "dataproxy");

    TFakeMetabaseNode nodeB{"node-b"};
    auto b1 = nodeB.AddShard(100, "yql", "prestable", "zookeeper");
    auto b2 = nodeB.AddShard(200, "yasm", "production", "histdb");

    auto rpc = MakeFakeRpc({&nodeA, &nodeB});
    auto watcher = MetabaseClusterWatcher(rpc, rpc->Addresses(), TDuration::Seconds(1));
    auto watcherId = ActorRuntime_->Register(watcher.release());
    ActorRuntime_->WaitForBootstrap();

    ActorRuntime_->Send(watcherId, EdgeId_, MakeHolder<TMetabaseWatcherEvents::TSubscribe>());

    {
        auto change = ReceiveResponse<TMetabaseWatcherEvents::TStateChanged>();
        ASSERT_TRUE(change);

        const auto& updated = change->Get()->Updated;
        ASSERT_EQ(updated.size(), 3u);

        if (auto it = FindShard(updated, 1u); it != updated.end()) {
            EXPECT_EQ((*it)->Id, 1u);
            EXPECT_EQ((*it)->Key, a1->Key);
            EXPECT_EQ((*it)->Address, nodeA.Address());
        } else {
            FAIL() << "shard with id=1 not found";
        }

        if (auto it = FindShard(updated, 100u); it != updated.end()) {
            EXPECT_EQ((*it)->Id, 100u);
            EXPECT_EQ((*it)->Key, b1->Key);
            EXPECT_EQ((*it)->Address, nodeB.Address());
        } else {
            FAIL() << "shard with id=100 not found";
        }

        if (auto it = FindShard(updated, 200u); it != updated.end()) {
            EXPECT_EQ((*it)->Id, 200u);
            EXPECT_EQ((*it)->Key, b2->Key);
            EXPECT_EQ((*it)->Address, nodeB.Address());
        } else {
            FAIL() << "shard with id=200 not found";
        }

        EXPECT_EQ(change->Get()->Removed.size(), 0u);
    }

    // add new shard on node A
    auto a2 = nodeA.AddShard(2, "solomon", "testing", "coremon");
    ActorRuntime_->AdvanceCurrentTime(TDuration::Seconds(10));

    {
        auto change = ReceiveResponse<TMetabaseWatcherEvents::TStateChanged>();
        ASSERT_TRUE(change);

        const auto& updated = change->Get()->Updated;
        ASSERT_EQ(updated.size(), 1u);

        EXPECT_EQ(updated[0]->Id, 2u);
        EXPECT_EQ(updated[0]->Key, a2->Key);
        EXPECT_EQ(updated[0]->Address, nodeA.Address());

        EXPECT_EQ(change->Get()->Removed.size(), 0u);
    }

    // move shard 100 from node B to node A
    b1 = nodeB.RemoveShard(100);
    nodeA.AddShard(b1->Id, b1->Key.Project, b1->Key.SubKey.Cluster, b1->Key.SubKey.Service);
    ActorRuntime_->AdvanceCurrentTime(TDuration::Seconds(100));

    {
        auto change = ReceiveResponse<TMetabaseWatcherEvents::TStateChanged>();
        ASSERT_TRUE(change);

        const auto& removed = change->Get()->Removed;
        if (!removed.empty()) {
            // if watcher receives response from a node where shard was removed before
            // response from a node where shard was added then it send two TStateChanged events
            EXPECT_EQ(removed.size(), 1u);
            EXPECT_EQ(removed[0], 100u);

            // get second change event from watcher
            change = ReceiveResponse<TMetabaseWatcherEvents::TStateChanged>();
            ASSERT_TRUE(change);
            ASSERT_TRUE(change->Get()->Removed.empty());
        }

        // otherwise watcher merges shard add and remove events from nodes and send only one
        // TStateChanged event
        const auto& updated = change->Get()->Updated;
        ASSERT_EQ(updated.size(), 1u);

        EXPECT_EQ(updated[0]->Id, 100u);
        EXPECT_EQ(updated[0]->Key, b1->Key);
        EXPECT_EQ(updated[0]->Address, nodeA.Address()); // moved to node A
    }

    // remove shard 200 from node B
    nodeB.RemoveShard(200);
    ActorRuntime_->AdvanceCurrentTime(TDuration::Seconds(10));

    {
        auto update = ReceiveResponse<TMetabaseWatcherEvents::TStateChanged>();
        ASSERT_TRUE(update);

        EXPECT_EQ(update->Get()->Updated.size(), 0u);

        auto shardIds = update->Get()->Removed;
        EXPECT_EQ(shardIds.size(), 1u);
        EXPECT_EQ(shardIds[0], 200u);
    }
}

TEST_F(TMetabaseWatcherTest, Resolve) {
    TFakeMetabaseNode nodeA{"node-a"};
    auto a1 = nodeA.AddShard(1, "solomon", "production", "dataproxy");

    TFakeMetabaseNode nodeB{"node-b"};
    auto b1 = nodeB.AddShard(100, "yql", "prestable", "zookeeper");
    auto b2 = nodeB.AddShard(200, "yasm", "production", "histdb");

    auto rpc = MakeFakeRpc({&nodeA, &nodeB});
    auto watcher = MetabaseClusterWatcher(rpc, rpc->Addresses(), TDuration::Seconds(1));
    auto watcherId = ActorRuntime_->Register(watcher.release());
    ActorRuntime_->WaitForBootstrap();

    {
        auto event = std::make_unique<TMetabaseWatcherEvents::TResolve>();
        event->Ids.push_back(100);
        event->Ids.push_back(1);
        ActorRuntime_->Send(watcherId, EdgeId_, THolder(event.release()));

        auto result = ReceiveResponse<TMetabaseWatcherEvents::TResolveResult>();
        ASSERT_TRUE(result);

        const auto& locations = result->Get()->Locations;
        EXPECT_EQ(locations.size(), 2u);

        {
            auto it = FindShard(locations, 1u);
            ASSERT_TRUE(it != locations.end());
            EXPECT_EQ(it->Id, 1u);
            EXPECT_EQ(it->Address, "node-a");
        }

        {
            auto it = FindShard(locations, 100u);
            ASSERT_TRUE(it != locations.end());
            EXPECT_EQ(it->Id, 100u);
            EXPECT_EQ(it->Address, "node-b");
        }
    }

    {
        auto event = std::make_unique<TMetabaseWatcherEvents::TResolve>();
        event->Ids.push_back(200);
        event->Ids.push_back(300);
        ActorRuntime_->Send(watcherId, EdgeId_, THolder(event.release()));

        auto result = ReceiveResponse<TMetabaseWatcherEvents::TResolveResult>();
        ASSERT_TRUE(result);

        const auto& locations = result->Get()->Locations;
        EXPECT_EQ(locations.size(), 1u);

        {
            auto it = FindShard(locations, 200u);
            ASSERT_TRUE(it != locations.end());
            EXPECT_EQ(it->Id, 200u);
            EXPECT_EQ(it->Address, "node-b");
        }
    }
}
