#pragma once

#include <solomon/services/dataproxy/lib/limits.h>
#include <solomon/services/dataproxy/lib/cluster_id/dc.h>
#include <solomon/services/dataproxy/lib/cluster_id/replica_map.h>

#include <library/cpp/monlib/metrics/metric_type.h>

#include <util/generic/vector.h>
#include <util/digest/numeric.h>

#include <array>

namespace NSolomon::NDataProxy {

template <typename T>
struct TLabel {
    T Key{};
    T Value{};

    TLabel() = default;

    TLabel(T key, T value) noexcept
        : Key(std::move(key))
        , Value(std::move(value))
    {
    }

    bool operator==(const TLabel<T>& rhs) const noexcept {
        return Key == rhs.Key && Value == rhs.Value;
    }

    bool operator!=(const TLabel<T>& rhs) const noexcept {
        return !(*this == rhs);
    }

    size_t Hash() const noexcept {
        return CombineHashes(std::hash<T>{}(Key), std::hash<T>{}(Value));
    }

    bool operator<(const TLabel<T>& rhs) const noexcept {
        return Key == rhs.Key ? Value < rhs.Value : Key < rhs.Key;
    }
};

template <typename T>
using TLabels = TVector<TLabel<T>>;

template <typename T>
struct TMetricKey {
    T Name{};
    TLabels<T> Labels;

    bool operator==(const TMetricKey<T>& rhs) const noexcept {
        return Name == rhs.Name && Labels == rhs.Labels;
    }

    bool operator!=(const TMetricKey<T>& rhs) const noexcept {
        return !(*this == rhs);
    }

    size_t Hash() const noexcept {
        return CombineHashes(std::hash<T>{}(Name), std::hash<decltype(Labels)>{}(Labels));
    }
};

struct TStockpileId {
    ui32 ShardId{0};
    ui64 LocalId{0};

    bool operator==(const TStockpileId& rhs) const noexcept {
        return ShardId == rhs.ShardId && LocalId == rhs.LocalId;
    }

    bool operator!=(const TStockpileId& rhs) const noexcept {
        return !(*this == rhs);
    }

    size_t Hash() const noexcept {
        return CombineHashes(static_cast<ui64>(ShardId), LocalId);
    }
};

template <typename T>
struct TMetric {
    NMonitoring::EMetricType Type{};
    T Name{};
    TLabels<T> Labels;
    // TODO: will be dropped, as soon as DataProxy learns to work with Stockpile
    TReplicaMap<TStockpileId> StockpileIds;

    bool operator==(const TMetric<T>& rhs) const noexcept {
        return Type == rhs.Type && Name == rhs.Name && Labels == rhs.Labels && StockpileIds == rhs.StockpileIds;
    }

    bool operator!=(const TMetric<T>& rhs) const noexcept {
        return !(*this == rhs);
    }

    size_t Hash() const noexcept {
        size_t hash = CombineHashes(static_cast<size_t>(Type), std::hash<T>{}(Name));
        hash = CombineHashes(hash, std::hash<decltype(Labels)>{}(Labels));
        hash = CombineHashes(hash, StockpileIds.Hash());
        return hash;
    }
};

} // namespace NSolomon::NDataProxy

template <typename T>
struct std::hash<NSolomon::NDataProxy::TLabel<T>> {
    size_t operator()(const NSolomon::NDataProxy::TLabel<T>& l) const {
        return l.Hash();
    }
};

template <typename T>
struct std::hash<NSolomon::NDataProxy::TLabels<T>> {
    size_t operator()(const NSolomon::NDataProxy::TLabels<T>& labels) const {
        size_t hash = 0;
        for (const auto& l: labels) {
            hash = CombineHashes(hash, l.Hash());
        }
        return hash;
    }
};

template <typename T>
struct std::hash<NSolomon::NDataProxy::TMetricKey<T>> {
    size_t operator()(const NSolomon::NDataProxy::TMetricKey<T>& mk) const {
        return mk.Hash();
    }
};

template <typename T>
struct std::hash<NSolomon::NDataProxy::TMetric<T>> {
    size_t operator()(const NSolomon::NDataProxy::TMetric<T>& m) const {
        return m.Hash();
    }
};
