#include "metrics_page.h"
#include "events.h"
#include "tracking.h"
#include "rendering.h"

#include <solomon/services/dataproxy/lib/datasource/merge_sts_lts/merge_sts_lts.h>
#include <solomon/services/dataproxy/lib/datasource/yasm/dynamic_aggregation.h>
#include <solomon/services/dataproxy/lib/datasource/yasm/yasm.h>
#include <solomon/libs/cpp/error_or/error_or.h>
#include <solomon/libs/cpp/selfmon/selfmon.h>
#include <solomon/libs/cpp/string_map/string_map.h>

#include <library/cpp/actors/core/actor_bootstrapped.h>
#include <library/cpp/string_utils/quote/quote.h>

using namespace NActors;
using namespace NSolomon::NSelfMon;
using namespace yandex::monitoring::selfmon;

namespace NSolomon::NDataProxy {

namespace {

TActorId GetMetricsPageId() {
    return TActorId(0, "METRICS_PAGE");
}

class TDummyHandler: public IResultHandler<TReadManyResult> {
private:
    void OnSuccess(std::unique_ptr<TReadManyResult>) override {
    }
    void OnError(TString&&, EDataSourceStatus, TString&&) override {
    }
};

class TMetricsPage: public TActorBootstrapped<TMetricsPage> {
public:
    explicit TMetricsPage(IDataSourcePtr dataSource)
        : DataSource_{std::move(dataSource)}
    {
    }

    using TEventVariant =
            std::variant<TFindQuery, TFindResult, TReadManyQuery, TReadManyResult, TLabelValuesQuery, TLabelValuesResult, TDataSourceError>;
    STATEFN(StateFunc) {
        switch (ev->GetTypeRewrite()) {
            hFunc(TEvPageDataReq, OnRequest)
            hFunc(TMetricsTracking::TMessage<TFindQuery>, OnTrackingMessage<TFindQuery>)
            hFunc(TMetricsTracking::TMessage<TFindResult>, OnTrackingMessage<TFindResult>)
            hFunc(TMetricsTracking::TMessage<TReadManyQuery>, OnTrackingMessage<TReadManyQuery>)
            hFunc(TMetricsTracking::TMessage<TReadManyResult>, OnTrackingMessage<TReadManyResult>)
            hFunc(TMetricsTracking::TMessage<TLabelValuesQuery>, OnTrackingMessage<TLabelValuesQuery>)
            hFunc(TMetricsTracking::TMessage<TLabelValuesResult>, OnTrackingMessage<TLabelValuesResult>)
            hFunc(TMetricsTracking::TMessage<TDataSourceError>, OnTrackingMessage<TDataSourceError>)
            hFunc(NActors::TEvents::TEvPoison, OnPoison)
        }
    }

    void Bootstrap() {
        TActorContext::ActorSystem()->RegisterLocalService(GetMetricsPageId(), SelfId());
        Become(&TMetricsPage::StateFunc);
    }

private:
    void OnRequest(TEvPageDataReq::TPtr& ev) {
        Page page;
        auto* grid = page.mutable_grid();
        if (auto stage = ev->Get()->Param("stage")) {
            size_t eventInd;
            if (auto eventIndStr = ev->Get()->Param("eventInd"); TryFromString(eventIndStr, eventInd)) {
                if (auto it = StageToEvents_.find(stage); it != StageToEvents_.end() && eventInd < it->second.size()) {
                    try {
                        std::visit([grid, &ev, &stage](auto&& event) { RenderHeader(event, stage, grid); RenderEvent(event, grid, ev->Get()); }, it->second[eventInd]);
                    } catch (const std::exception& exc) {
                        auto* h = grid->add_rows()->add_columns()->mutable_component()->mutable_heading();
                        h->set_level(3);
                        h->set_content(FormatExc(exc));
                    }
                } else {
                    auto* h = grid->add_rows()->add_columns()->mutable_component()->mutable_heading();
                    h->set_level(3);
                    h->set_content(TStringBuilder{} << "NOT FOUND: \"" << stage << "\":" << eventInd);
                }
            }
            Send(ev->Sender, new TEvPageDataResp{std::move(page)});
            return;
        }

        if (ev->Get()->HttpReq->Method == "POST") {
            StageToEvents_.clear();
            TErrorOr<TSelectors, TInvalidSelectorsFormat> selectors = TSelectors{};
            TQuickCgiParam params{ev->Get()->HttpReq->Body};
            if (auto selectorsStr = TString{params.Get("selectors")}) {
                CGIUnescape(selectorsStr);
                if (!selectorsStr.StartsWith('{')) {
                    selectorsStr = "{" + selectorsStr + "}";
                }
                Selectors_ = selectorsStr;
                try {
                    selectors = ParseSelectors(selectorsStr);
                } catch (const TInvalidSelectorsFormat& err) {
                    selectors = err;
                }
                if (selectors.Success()) {
                    TReadManyQuery query;
                    query.Project = selectors.Value().Find("project")->Pattern();
                    auto now = TActivationContext::Now();
                    query.Time = {now - TDuration::Minutes(30), now};
                    query.Lookup = std::make_unique<TReadManyQuery::TLookup>(selectors.Extract(), 10'000);
                    query.Deadline = now + TDuration::Seconds(30);
                    query.MaxTimeSeriesFormat = 39;

                    yandex::solomon::math::Operation operation;
                    operation.mutable_downsampling()->set_grid_millis(5'000);
                    query.Operations.push_back(std::move(operation));

                    DataSource_->ReadMany(std::move(query), MakeIntrusive<TDummyHandler>());
                } else {
                    auto* r = grid->add_rows();
                    auto* c = r->add_columns()->mutable_component();
                    auto* h = c->mutable_heading();
                    h->set_level(3);
                    h->set_content(FormatExc(selectors.Error()));
                }
            }
        }

        {
            auto* r = grid->add_rows();
            auto* form = r->add_columns()->mutable_component()->mutable_form();
            form->set_method(FormMethod::Post);

            auto* item = form->add_items();
            item->set_label("Selectors");
            item->set_help("range [now - 30m, now], limit 10'000, downsampling 5s");

            auto* input = item->mutable_input();
            input->set_type(InputType::Text);
            input->set_value(Selectors_);
            input->set_name("selectors");

            auto* submit = form->add_submit();
            submit->set_title("Execute");
        }

        RenderStages(grid, ev->Get());

        Send(ev->Sender, new TEvPageDataResp{std::move(page)});
    }

    template <typename TEvent>
    void OnTrackingMessage(typename TMetricsTracking::TMessage<TEvent>::TPtr& ev) {
        StageToEvents_[ev->Get()->Stage].emplace_back(std::move(ev->Get()->Payload));
    }

    void OnPoison(NActors::TEvents::TEvPoison::TPtr& ev) {
        Send(ev->Sender, new NActors::TEvents::TEvPoisonTaken);
        PassAway();
    }

    void RenderStages(Grid* grid, TEvPageDataReq* req) {
        for (auto&& [stage, values]: StageToEvents_) {
            auto* h = grid->add_rows()->add_columns()->mutable_component()->mutable_heading();
            h->set_level(4);
            h->set_content(stage);
            for (size_t i = 0; i < values.size(); ++i) {
                auto* ref = grid->add_rows()->add_columns()->mutable_component()->mutable_value()->mutable_reference();
                ref->set_title(std::visit([](auto&& arg) { return EventDecription(arg); }, values[i]));
                ref->set_page("/metrics");
                ref->set_args(TString{req->Query} + "&stage=" + stage + "&eventInd=" + ToString(i));
            }
        }
    }

private:
    IDataSourcePtr DataSource_;
    TStringMap<TVector<TEventVariant>> StageToEvents_;
    TString Selectors_;
};

} // namespace

std::unique_ptr<IActor> CreateMetricsPage(
        TActorRuntime& runtime,
        TYasmConfig config,
        IDataSourcePtr shortTerm,
        IDataSourcePtr longTerm,
        NMonitoring::TMetricRegistry& registry)
{
    auto metricsPageId = GetMetricsPageId();
    auto& as = runtime.ActorSystem();
    auto trackedSts = TrackingDataSource(as, std::move(shortTerm), "STS", metricsPageId);
    auto trackedLts = TrackingDataSource(as, std::move(longTerm), "LTS", metricsPageId);
    auto merge = NMerger::MergeShortLongTermsSource(runtime, std::move(trackedSts), std::move(trackedLts));
    auto trackedMerge = TrackingDataSource(as, std::move(merge), "Merge", metricsPageId);
    auto aggr = YasmDynAggrDataSource(trackedMerge, &as);
    auto trackedAggr = TrackingDataSource(as, std::move(aggr), "DynAggr", metricsPageId);
    auto yasmDataSourceActor = runtime.Register(YasmDataSourceActor(std::move(config), trackedAggr, registry));
    auto yasm = YasmDataSource(runtime, yasmDataSourceActor, std::move(trackedAggr));
    auto trackedYasm = TrackingDataSource(as, std::move(yasm), "Yasm", metricsPageId);
    return std::make_unique<TMetricsPage>(std::move(trackedYasm));
}

} // namespace NSolomon::NDataProxy
