#include "rendering.h"
#include "point_to_string.h"

#include <solomon/services/dataproxy/lib/datasource/comparison/out.h>
#include <solomon/libs/cpp/error_or/error_or.h>
#include <library/cpp/string_utils/quote/quote.h>
#include <util/string/vector.h>

using namespace NSolomon::NSelfMon;
using namespace yandex::monitoring::selfmon;

namespace NSolomon::NDataProxy {

namespace {

constexpr TStringBuf READMANYQUERY_RESOLVEDKEYS_METRICKEYS  = "ReadManyQuery.ResolvedKeys.MetricKeys";
constexpr TStringBuf FINDRESULT_METRICS = "FindResult.Metrics";
constexpr TStringBuf READMANYRESULT_METRICS = "ReadManyResult.Metrics";
constexpr TStringBuf LABELVALUESRESULT_LABELS = "LabelValuesResult.Labels";

TString LabelsToString(const TLabels<ui32>& labels, const NStringPool::TStringPool& strings) {
    TStringBuilder sb;
    sb << "{";
    bool first = true;
    for (auto [key, value]: labels) {
        if (!first) {
            sb << ", ";
        }
        first = false;

        sb << strings.At(key) << "=\"" << strings.At(value) << '"';
    }
    sb << "}";
    return sb;
}

TString DataSourceErrorsToString(const TDataSourceErrors& errors) {
    TStringBuilder sb;
    bool first = true;
    for (auto&& error: errors) {
        if (!first) {
            sb << ", ";
        }
        first = false;
        sb << error.Status << ": " << error.Message;
    }
    return sb;
}

void RenderMetricKeys(
        const TVector<TMetricKey<ui32>>& keys,
        const NStringPool::TStringPool& strings,
        const TSelectors&,
        Grid* grid)
{
    auto* t = grid->add_rows()->add_columns()->mutable_component()->mutable_table();
    t->set_numbered(true);

    auto* nameColumn = t->add_columns();
    nameColumn->set_title("Name");
    auto* nameValues = nameColumn->mutable_string();

    auto* labelsColumn = t->add_columns();
    labelsColumn->set_title("Labels");
    auto* labelsValues = labelsColumn->mutable_string();

    for (size_t i = 0; i < keys.size(); ++i) {
        const auto& key = keys[i];
        nameValues->add_values(TString{strings[key.Name]});
        labelsValues->add_values(LabelsToString(key.Labels, strings));
    }
}

TErrorOr<TSelectors, TInvalidSelectorsFormat> SelectorsFromReq(TEvPageDataReq* req) {
    TErrorOr<TSelectors, TInvalidSelectorsFormat> selectors = TSelectors{};
    auto selectorsStr = TString{req->Param("selectors")};
    if (selectorsStr) {
        CGIUnescape(selectorsStr);
        if (!selectorsStr.StartsWith('{')) {
            selectorsStr = "{" + selectorsStr + "}";
        }
        try {
            selectors = ParseSelectors(selectorsStr);
        } catch (TInvalidSelectorsFormat& err) {
            selectors = err;
        }
    }

    return selectors;
}

void AddFilterForm(Grid* grid, TEvPageDataReq* req) {
    auto selectorsStr = TString{req->Param("selectors")};
    if (selectorsStr) {
        CGIUnescape(selectorsStr);
        if (!selectorsStr.StartsWith('{')) {
            selectorsStr = "{" + selectorsStr + "}";
        }
    }

    auto* form = grid->add_rows()->add_columns()->mutable_component()->mutable_form();
    form->set_method(yandex::monitoring::selfmon::FormMethod::Get);

    {
        auto* input = form->add_items()->mutable_input();
        input->set_type(yandex::monitoring::selfmon::InputType::Hidden);
        input->set_name("stage");
        input->set_value(TString(req->Param("stage")));
    }

    {
        auto* input = form->add_items()->mutable_input();
        input->set_type(yandex::monitoring::selfmon::InputType::Hidden);
        input->set_name("eventInd");
        input->set_value(TString(req->Param("eventInd")));
    }

    {
        auto* item = form->add_items();
        item->set_label("Filter");
        auto* input = item->mutable_input();
        input->set_type(yandex::monitoring::selfmon::InputType::Text);
        input->set_name("selectors");
        input->set_value(TString{req->Param("selectors")});
    }

    auto* submit = form->add_submit();
    submit->set_title("Filter");
}

void RenderMetricsData(
        const TVector<TReadManyResult::TMetricData>& metrics,
        const NStringPool::TStringPool& strings,
        const TSelectors&,
        Grid* grid,
        TEvPageDataReq* req)
{
    auto* t = grid->add_rows()->add_columns()->mutable_component()->mutable_table();
    t->set_numbered(true);

    auto* typeColumn = t->add_columns();
    typeColumn->set_title("Type");
    auto* typeValue = typeColumn->mutable_string();

    auto* nameColumn = t->add_columns();;
    nameColumn->set_title("Name");
    auto* nameValues = nameColumn->mutable_string();

    auto* labelsColumn = t->add_columns();;
    labelsColumn->set_title("Labels");
    auto* labelsValues = labelsColumn->mutable_string();

    auto* tsColumn = t->add_columns();;
    tsColumn->set_title("TimeSeries");
    auto* tsValues = tsColumn->mutable_reference();

    for (size_t i = 0; i < metrics.size(); ++i) {
        const auto& metric = metrics[i];
        typeValue->add_values(TString{NMonitoring::MetricTypeToStr(metric.Meta.Type)});
        nameValues->add_values(TString{strings[metric.Meta.Name]});
        labelsValues->add_values(LabelsToString(metric.Meta.Labels, strings));

        auto* r = tsValues->add_values();
        r->set_page("/metrics");
        r->set_args(TStringBuilder{} << req->Query << "&tsInd=" << i);
        r->set_title(ToString(metric.TimeSeries->PointCount()) + " points");
    }
}

void RenderMetrics(const TVector<TMetric<ui32>>& metrics, 
                   const NStringPool::TStringPool& strings,
                   const TSelectors&,
                   Grid* grid)
{
    auto* t = grid->add_rows()->add_columns()->mutable_component()->mutable_table();
    t->set_numbered(true);

    auto* typeColumn = t->add_columns();
    typeColumn->set_title("Type");
    auto* typeValue = typeColumn->mutable_string();

    auto* nameColumn = t->add_columns();;
    nameColumn->set_title("Name");
    auto* nameValues = nameColumn->mutable_string();

    auto* labelsColumn = t->add_columns();;
    labelsColumn->set_title("Labels");
    auto* labelsValues = labelsColumn->mutable_string();

    auto* stockpileIdsColumn = t->add_columns();;
    stockpileIdsColumn->set_title("StockpileIds");
    auto* stockpileIdsValues = stockpileIdsColumn->mutable_string();

    for (size_t i = 0; i < metrics.size(); ++i) {
        const auto& metric = metrics[i];
        typeValue->add_values(TString{NMonitoring::MetricTypeToStr(metric.Type)});
        nameValues->add_values(TString{strings[metric.Name]});
        labelsValues->add_values(LabelsToString(metric.Labels, strings));
        TStringBuilder sb;
        for (auto replica: KnownReplicas) {
            sb << "{" << ReplicaToStr(replica) << ": [" << metric.StockpileIds[replica].ShardId << ", "
               << metric.StockpileIds[replica].LocalId << "]} ";
        }
        stockpileIdsValues->add_values(sb);
    }
}

void RenderTimeSeries(const std::unique_ptr<ITimeSeries>& ts, Grid* grid) {
    auto* t = grid->add_rows()->add_columns()->mutable_component()->mutable_table();
    t->set_numbered(true);

    auto timeColumn = t->add_columns();
    timeColumn->set_title("Time");
    auto* timeValues = timeColumn->mutable_time();

    auto valueColumn = t->add_columns();
    valueColumn->set_title("Value");
    auto* valueValues = valueColumn->mutable_string();

    auto stepColumn = t->add_columns();
    stepColumn->set_title("Step");
    auto* stepValues = stepColumn->mutable_duration();

    auto countColumn = t->add_columns();
    countColumn->set_title("Count");
    auto* countValues = countColumn->mutable_uint64();

    auto mergeColumn = t->add_columns();
    mergeColumn->set_title("Merge");
    auto* mergeValues = mergeColumn->mutable_boolean();

    auto iter = ts->Iterator();
    NTs::TVariantPoint point;
    while (iter->Next(&point)) {
        timeValues->add_values(point.Time.GetValue());
        stepValues->add_values(point.Step.GetValue());
        countValues->add_values(point.Count);
        mergeValues->add_values(point.Merge);
        valueValues->add_values(PointToString(point));
    }
}

} // namespace

void AddQueryFieldsBase(const TQuery& query, Object* o) {
    {
        auto* f = o->add_fields();
        f->set_name("Project");
        f->mutable_value()->set_string(query.Project);
    }
    {
        auto* f = o->add_fields();
        f->set_name("Time.From");
        f->mutable_value()->set_time(query.Time.From.GetValue());
    }
    {
        auto* f = o->add_fields();
        f->set_name("Time.To");
        f->mutable_value()->set_time(query.Time.To.GetValue());
    }
    {
        auto* f = o->add_fields();
        f->set_name("Deadline");
        f->mutable_value()->set_time(query.Deadline.GetValue());
    }
    if (query.SoftDeadline) {
        auto* f = o->add_fields();
        f->set_name("SoftDeadline");
        f->mutable_value()->set_time(query.SoftDeadline->GetValue());
    }
    {
        auto* f = o->add_fields();
        f->set_name("ForceReplicaRead");
        f->mutable_value()->set_string(query.ForceReplicaRead);
    }
    {
        auto* f = o->add_fields();
        f->set_name("Producer");
        f->mutable_value()->set_string(RequestProducer_Name(query.Producer));
    }
    {
        auto* f = o->add_fields();
        f->set_name("ShortTermStorage");
        f->mutable_value()->set_string(yandex::monitoring::dataproxy::ShortTermStorage_Name(query.ShortTermStorage));
    }
    {
        auto* f = o->add_fields();
        f->set_name("ClientId");
        f->mutable_value()->set_string(TString{query.ClientId});
    }
}

void AddQueryFields(const TFindQuery& query, Object* o, TEvPageDataReq*) {
    {
        auto* f = o->add_fields();
        f->set_name("Selectors");
        f->mutable_value()->set_string(ToString(query.Selectors));
    }
    {
        auto* f = o->add_fields();
        f->set_name("Limit");
        f->mutable_value()->set_uint32(query.Limit);
    }
    {
        auto* f = o->add_fields();
        f->set_name("FillMetricName");
        f->mutable_value()->set_boolean(query.FillMetricName);
    }
}

void AddQueryFields(const TReadManyQuery& query, Object* o, TEvPageDataReq* req) {
    if (query.ResolvedKeys) {
        {
            auto* f = o->add_fields();
            f->set_name("ResolvedKeys.CommonLabels");
            f->mutable_value()->set_string(LabelsToString(query.ResolvedKeys->CommonLabels, query.ResolvedKeys->Strings));
        }
        {
            auto* f = o->add_fields();
            f->set_name("ResolvedKeys.MetricKeys");
            auto* r = f->mutable_value()->mutable_reference();
            r->set_title(ToString(query.ResolvedKeys->MetricKeys.size()));
            r->set_page("/metrics");
            r->set_args(TString{req->Query} + "&field=" + READMANYQUERY_RESOLVEDKEYS_METRICKEYS);
        }
    }
    if (query.Lookup) {
        {
            auto* f = o->add_fields();
            f->set_name("Lookup.Selectors");
            f->mutable_value()->set_string(ToString(query.Lookup->Selectors));
        }
        {
            auto* f = o->add_fields();
            f->set_name("Lookup.Limit");
            f->mutable_value()->set_uint32(query.Lookup->Limit);
        }
        {
            auto* f = o->add_fields();
            f->set_name("Lookup.GroupBy");
            f->mutable_value()->set_string(JoinStrings(query.Lookup->GroupBy, ", "));
        }
    }

    {
        auto* f = o->add_fields();
        f->set_name("MaxTimeSeriesFormat");
        f->mutable_value()->set_uint32(query.MaxTimeSeriesFormat);
    }

    {
        auto* f = o->add_fields();
        f->set_name("Operations");
        TStringBuilder sb;
        for (const auto& op: query.Operations) {
            sb << op.ShortUtf8DebugString() << " ";
        }
        f->mutable_value()->set_string(sb);
    }
}

void AddQueryFields(const TLabelValuesQuery& query, Object* o, TEvPageDataReq*) {
    {
        auto* f = o->add_fields();
        f->set_name("Selectors");
        f->mutable_value()->set_string(ToString(query.Selectors));
    }
    {
        auto* f = o->add_fields();
        f->set_name("Keys");
        f->mutable_value()->set_string(JoinStrings(query.Keys, ", "));
    }
    {
        auto* f = o->add_fields();
        f->set_name("TextFilter");
        f->mutable_value()->set_string(query.TextFilter);
    }
    {
        auto* f = o->add_fields();
        f->set_name("Limit");
        f->mutable_value()->set_uint32(query.Limit);
    }
}


void AddResultFields(const TReadManyResult& result, Object* o, TEvPageDataReq* req) {
    {
        auto* f = o->add_fields();
        f->set_name("Metrics");
        auto* r = f->mutable_value()->mutable_reference();
        r->set_title(ToString(result.Metrics.size()));
        r->set_page("/metrics");
        r->set_args(TString{req->Query} + "&field=" + READMANYRESULT_METRICS);
    }
    {
        auto* f = o->add_fields();
        f->set_name("Errors");
        f->mutable_value()->set_string(DataSourceErrorsToString(result.Errors));
    }
}

void AddResultFields(const TFindResult& result, Object* o, TEvPageDataReq* req) {
    {
        auto* f = o->add_fields();
        f->set_name("Metrics");
        auto* r = f->mutable_value()->mutable_reference();
        r->set_title(ToString(result.Metrics.size()));
        r->set_page("/metrics");
        r->set_args(TString{req->Query} + "&field=" + FINDRESULT_METRICS);
    }
    {
        auto* f = o->add_fields();
        f->set_name("Errors");
        f->mutable_value()->set_string(DataSourceErrorsToString(result.Errors));
    }
    {
        auto* f = o->add_fields();
        f->set_name("TotalCount");
        f->mutable_value()->set_uint32(result.TotalCount);
    }
    {
        auto* f = o->add_fields();
        f->set_name("Truncated");
        f->mutable_value()->set_boolean(result.Truncated);
    }

    TStringBuilder sb;
    for (auto replica: KnownReplicas) {
        sb << "{" << replica << ": " << result.Dcs[replica] << "} ";
    }
    {
        auto* f = o->add_fields();
        f->set_name("Dcs");
        f->mutable_value()->set_string(sb);
    }
}

void AddResultFields(const TLabelValuesResult& result, Object* o, TEvPageDataReq* req) {
    {
        auto* f = o->add_fields();
        f->set_name("Project");
        f->mutable_value()->set_string(result.Project);
    }
    {
        auto* f = o->add_fields();
        f->set_name("Labels");
        auto* r = f->mutable_value()->mutable_reference();
        r->set_title(ToString(result.Labels.size()));
        r->set_page("/metrics");
        r->set_args(TString{req->Query} + "&field=" + LABELVALUESRESULT_LABELS);
    }
    {
        auto* f = o->add_fields();
        f->set_name("Errors");
        f->mutable_value()->set_string(DataSourceErrorsToString(result.Errors));
    }
    {
        auto* f = o->add_fields();
        f->set_name("MetricCount");
        f->mutable_value()->set_uint32(result.MetricCount);
    }
}

void RenderField(const TReadManyResult& result, Grid* grid, TEvPageDataReq* req) {
    Y_ENSURE(req->Param("field") == READMANYRESULT_METRICS);

    size_t tsInd;
    if (TString tsIndStr = TString{req->Param("tsInd")}; TryFromString(tsIndStr, tsInd)) {
        RenderTimeSeries(result.Metrics[tsInd].TimeSeries, grid);
        return;
    }

    AddFilterForm(grid, req);
    auto selectors = SelectorsFromReq(req);

    {
        auto* h = grid->add_rows()->add_columns()->mutable_component()->mutable_heading();
        if (selectors.Success()) {
            h->set_content("ReadManyResult.Metrics");
        } else {
            h->set_content(FormatExc(selectors.Error()));
        }
        h->set_level(4);
    }

    if (selectors.Success()) {
        RenderMetricsData(result.Metrics, result.Strings.Copy().Build(), selectors.Value(), grid, req);
    }
}

void RenderField(const TReadManyQuery& query, Grid* grid, TEvPageDataReq* req) {
    Y_ENSURE(req->Param("field") == READMANYQUERY_RESOLVEDKEYS_METRICKEYS);
    Y_ENSURE(query.ResolvedKeys);

    AddFilterForm(grid, req);
    auto selectors = SelectorsFromReq(req);

    {
        auto* h = grid->add_rows()->add_columns()->mutable_component()->mutable_heading();
        if (selectors.Success()) {
            h->set_content(ToString(READMANYQUERY_RESOLVEDKEYS_METRICKEYS));
        } else {
            h->set_content(FormatExc(selectors.Error()));
        }
        h->set_level(4);
    }

    if (selectors.Success()) {
        RenderMetricKeys(query.ResolvedKeys->MetricKeys, query.ResolvedKeys->Strings, selectors.Value(), grid);
    }
}

void RenderField(const TFindResult& result, Grid* grid, TEvPageDataReq* req) {
    Y_ENSURE(req->Param("field") == FINDRESULT_METRICS);

    AddFilterForm(grid, req);
    auto selectors = SelectorsFromReq(req);

    {
        auto* h = grid->add_rows()->add_columns()->mutable_component()->mutable_heading();
        if (selectors.Success()) {
            h->set_content(ToString(FINDRESULT_METRICS));
        } else {
            h->set_content(FormatExc(selectors.Error()));
        }
        h->set_level(4);
    }

    if (selectors.Success()) {
        RenderMetrics(result.Metrics, result.Strings.Copy().Build(), selectors.Value(), grid);
    }
}

void RenderField(const TLabelValuesResult& result, Grid* grid, TEvPageDataReq* req) {
    Y_ENSURE(req->Param("field") == LABELVALUESRESULT_LABELS);
    auto strings = result.Strings.Copy().Build();

    auto* t = grid->add_rows()->add_columns()->mutable_component()->mutable_table();

    auto* keyColumn = t->add_columns();
    keyColumn->set_title("Key");
    auto* keyValues = keyColumn->mutable_string();

    auto* metricCountColumn = t->add_columns();
    metricCountColumn->set_title("MetricCount");
    auto* metricCountValues = metricCountColumn->mutable_uint32();

    auto* truncatedColumn = t->add_columns();
    truncatedColumn->set_title("Truncated");
    auto* truncatedValues = truncatedColumn->mutable_boolean();

    auto* valuesColumn = t->add_columns();
    valuesColumn->set_title("Values");
    auto* valuesValues = valuesColumn->mutable_string();

    auto* errorsColumn = t->add_columns();
    errorsColumn->set_title("Errors");
    auto* errorsValues = errorsColumn->mutable_string();

    for (auto&& label: result.Labels) {
        keyValues->add_values(TString{strings[label.Key]});
        metricCountValues->add_values(label.MetricCount);
        truncatedValues->add_values(label.Truncated);
        TStringBuilder sb;
        bool first = true;
        for (auto v: label.Values) {
            if (!first) {
                sb << ", ";
            } else {
                first = false;
            }
            sb << strings[v];
        }
        valuesValues->add_values(sb);
        errorsValues->add_values(DataSourceErrorsToString(label.Errors));
    }
}

void RenderField(const TFindQuery&, Grid*, TEvPageDataReq*) {
}

void RenderField(const TLabelValuesQuery&, Grid*, TEvPageDataReq*) {
}

} // namespace NSolomon::NDataProxy
