#include "tracking.h"
#include "events.h"

#include <solomon/services/dataproxy/lib/datasource/datasource.h>
#include <solomon/services/dataproxy/lib/timeseries/compressed.h>

#include <solomon/libs/cpp/actors/runtime/actor_runtime.h>
#include <library/cpp/actors/core/hfunc.h>

using namespace NActors;
using namespace NSolomon::NTracing;

namespace NSolomon::NDataProxy {
namespace {

template <typename TResult>
class TTrackingHandler: public IResultHandler<TResult> {
public:
    explicit TTrackingHandler(
            TActorSystem& actorSystem,
            IResultHandlerPtr<TResult> handler,
            TString stage,
            TActorId receiver)
        : ActorSystem_{actorSystem}
        , OriginalHandler_{std::move(handler)}
        , Stage_{std::move(stage)}
        , Receiver_{std::move(receiver)}
    {
    }

private:
    void OnSuccess(std::unique_ptr<TResult> result) override {
        ActorSystem_.Send(Receiver_, new TMetricsTracking::TMessage<TResult>{result->Clone(), Stage_});
        OriginalHandler_->OnSuccess(std::move(result));
    }

    void OnError(TString&& project, EDataSourceStatus status, TString&& message) override {
        ActorSystem_.Send(Receiver_, new TMetricsTracking::TMessage<TDataSourceError>{TDataSourceError{status, message}, Stage_});
        OriginalHandler_->OnError(std::move(project), status, std::move(message));
    }

private:
    TActorSystem& ActorSystem_;
    IResultHandlerPtr<TResult> OriginalHandler_;
    TString Stage_;
    TActorId Receiver_;
};

class TTrackingDataSource: public IDataSource {
public:
    TTrackingDataSource(TActorSystem& actorSystem, IDataSourcePtr baseDataSource, TString stage, TActorId receiver)
        : ActorSystem_{actorSystem}
        , BaseDataSource_{std::move(baseDataSource)}
        , Stage_{std::move(stage)}
        , Receiver_{std::move(receiver)}
    {
    }

private:
    bool CanHandle(const TQuery&) const override {
        Y_FAIL("Unexpected method call");
    }

    void ReadMany(TReadManyQuery query, IResultHandlerPtr<TReadManyResult> handler, TSpanId traceCtx) const override {
        ActorSystem_.Send(Receiver_, new TMetricsTracking::TMessage<TReadManyQuery>{query.Clone(), Stage_});
        handler = MakeIntrusive<TTrackingHandler<TReadManyResult>>(ActorSystem_, std::move(handler), Stage_, Receiver_);
        BaseDataSource_->ReadMany(std::move(query), std::move(handler), std::move(traceCtx));
    }

    void Find(TFindQuery query, IResultHandlerPtr<TFindResult> handler, TSpanId traceCtx) const override {
        ActorSystem_.Send(Receiver_, new TMetricsTracking::TMessage<TFindQuery>{query, Stage_});
        handler = MakeIntrusive<TTrackingHandler<TFindResult>>(ActorSystem_, std::move(handler), Stage_, Receiver_);
        BaseDataSource_->Find(std::move(query), std::move(handler), std::move(traceCtx));
    }

    void ResolveOne(TResolveOneQuery, IResultHandlerPtr<TResolveOneResult>, TSpanId) const override {
        Y_FAIL("Unexpected method call");
    }

    void ResolveMany(TResolveManyQuery, IResultHandlerPtr<TResolveManyResult>, TSpanId) const override {
        Y_FAIL("Unexpected method call");
    }

    void MetricNames(TMetricNamesQuery, IResultHandlerPtr<TMetricNamesResult>, TSpanId) const override {
        Y_FAIL("Unexpected method call");
    }

    void LabelKeys(TLabelKeysQuery, IResultHandlerPtr<TLabelKeysResult>, TSpanId) const override {
        Y_FAIL("Unexpected method call");
    }

    void LabelValues(TLabelValuesQuery query, IResultHandlerPtr<TLabelValuesResult> handler, TSpanId traceCtx) const override {
        ActorSystem_.Send(Receiver_, new TMetricsTracking::TMessage<TLabelValuesQuery>{query, Stage_});
        handler = MakeIntrusive<TTrackingHandler<TLabelValuesResult>>(ActorSystem_, std::move(handler), Stage_, Receiver_);
        BaseDataSource_->LabelValues(std::move(query), std::move(handler), std::move(traceCtx));
    }

    void UniqueLabels(TUniqueLabelsQuery, IResultHandlerPtr<TUniqueLabelsResult>, TSpanId) const override {
        Y_FAIL("Unexpected method call");
    }

    void ReadOne(TReadOneQuery, IResultHandlerPtr<TReadOneResult>, TSpanId) const override {
        Y_FAIL("Unexpected method call");
    }

    void WaitUntilInitialized() const override {
        Y_FAIL("Unexpected method call");
    }

private:
    TActorSystem& ActorSystem_;
    IDataSourcePtr BaseDataSource_;
    TString Stage_;
    TActorId Receiver_;
};

} // namespace

IDataSourcePtr TrackingDataSource(
        TActorSystem& actorSystem,
        IDataSourcePtr baseDataSource,
        TString stage,
        TActorId receiver)
{
    return MakeIntrusive<TTrackingDataSource>(actorSystem, std::move(baseDataSource), std::move(stage), std::move(receiver));
}

} // namespace NSolomon::NDataProxy
