#include "shard_selector.h"

namespace NSolomon::NDataProxy {
namespace {

static const TSelector ANY_CLUSTER{NLabels::LABEL_CLUSTER, AnyMatcher(), false};
static const TSelector ANY_SERVICE{NLabels::LABEL_SERVICE, AnyMatcher(), false};

TShardSelector MakeShardSelector(const TString& project, TStringBuf cluster, TStringBuf service) {
    Y_ENSURE(!cluster.empty(), "missed or empty 'cluster' label");
    Y_ENSURE(!service.empty(), "missed or empty 'service' label");

    TSelector clusterSelector{NLabels::LABEL_CLUSTER, ExactMatcher(cluster), false};
    TSelector serviceSelector{NLabels::LABEL_SERVICE, ExactMatcher(service), false};

    return TShardSelector{project, std::move(clusterSelector), std::move(serviceSelector)};
}

} // namespace

TShardSelector TShardSelector::FromPcs(TString project, TStringBuf cluster, TStringBuf service) {
    TSelector clusterSelector{NLabels::LABEL_CLUSTER, cluster, false};
    TSelector serviceSelector{NLabels::LABEL_SERVICE, service, false};
    return TShardSelector{std::move(project), std::move(clusterSelector), std::move(serviceSelector)};
}

TShardSelector TShardSelector::FromSelectors(const TString& project, const TSelectors& selectors) {
    auto clusterIt = selectors.Find(NLabels::LABEL_CLUSTER);
    TSelector clusterSelector = (clusterIt != selectors.cend()) ? *clusterIt : ANY_CLUSTER;

    auto serviceIt = selectors.Find(NLabels::LABEL_SERVICE);
    TSelector serviceSelector = (serviceIt != selectors.cend()) ? *serviceIt : ANY_SERVICE;

    return TShardSelector{project, std::move(clusterSelector), std::move(serviceSelector)};
}

TShardSelector TShardSelector::FromLabels(const TString& project, const TLabels<TString>& labels) {
    TStringBuf cluster, service;

    for (const auto& label: labels) {
        if (label.Key == NLabels::LABEL_CLUSTER) {
            cluster = label.Value;
        } else if (label.Key == NLabels::LABEL_SERVICE) {
            service = label.Value;
        }
    }

    return MakeShardSelector(project, cluster, service);
}

TShardSelector TShardSelector::FromLabels(
        const TString& project,
        const NStringPool::TStringPool& strings,
        const TLabels<ui32>& labels)
{
    TStringBuf cluster, service;

    for (const auto& label: labels) {
        auto key = strings[label.Key];
        if (key == NLabels::LABEL_CLUSTER) {
            cluster = strings[label.Value];
        } else if (key == NLabels::LABEL_SERVICE) {
            service = strings[label.Value];
        }
    }

    return MakeShardSelector(project, cluster, service);
}

IOutputStream& operator<<(IOutputStream& os, const TShardSelector& selector) {
    return os << TStringBuf{"{'project' == '"} << selector.Project
            << TStringBuf{"', "} << selector.Cluster
            << TStringBuf{", "} << selector.Service
            << '}';
}

} // namespace NSolomon::NDataProxy
