#pragma once

#include <solomon/services/dataproxy/lib/metric/metric.h>

#include <solomon/libs/cpp/labels/known_keys.h>
#include <solomon/libs/cpp/selectors/selectors.h>
#include <solomon/libs/cpp/string_pool/string_pool.h>

namespace NSolomon::NDataProxy {

struct TShardSelector {
    TString Project;
    TSelector Cluster;
    TSelector Service;

    TShardSelector() = default;

    // TODO: make it private
    TShardSelector(TString project, TSelector cluster, TSelector service) noexcept
        : Project(std::move(project))
        , Cluster(std::move(cluster))
        , Service(std::move(service))
    {
        Y_VERIFY(!Project.empty());
        Y_VERIFY(Cluster.Key() == NLabels::LABEL_CLUSTER, "invalid cluster selector key: '%s'", Cluster.Key().data());
        Y_VERIFY(Service.Key() == NLabels::LABEL_SERVICE, "invalid service selector key: '%s'", Service.Key().data());
    }

    /**
     * Create exact shard selector from given project/cluster/service.
     *
     * @param project  project id (must not be empty)
     * @param cluster  cluster name (must not be empty)
     * @param service  service name (must not be empty)
     * @return new shard selector object
     */
    static TShardSelector FromPcs(TString project, TStringBuf cluster, TStringBuf service);

    /**
     * Create shard selector from generic selectors.
     *
     * @param project      project id (must not be empty)
     * @param selectors    generic selectors (absence of cluster or service selector treated as any selector)
     * @return new shard selector object
     */
    static TShardSelector FromSelectors(const TString& project, const TSelectors& selectors);

    /**
     * Create shard selector from non interned labels.
     *
     * @param project      project id (must not be empty)
     * @param labels       non interned labels (must contain cluster and service labels)
     * @return new shard selector object
     */
    static TShardSelector FromLabels(const TString& project, const TLabels<TString>& labels);

    /**
     * Create shard selector from interned labels.
     *
     * @param project      project id (must not be empty)
     * @param strings      string pool which was used to intern given labels
     * @param labels       interned labels (must contain cluster and service labels)
     * @return new shard selector object
     */
    static TShardSelector FromLabels(
            const TString& project,
            const NStringPool::TStringPool& strings,
            const TLabels<ui32>& labels);

    /**
     * @return true iff shard selector uses exact matchers for cluster and service.
     */
    bool IsExact() const noexcept {
        return Cluster.IsExact() && Service.IsExact();
    }

    /**
     * Tries to match given cluster and service by current shard selector.
     *
     * @param cluster
     * @param service
     * @return
     */
    bool Match(TStringBuf cluster, TStringBuf service) const {
        return Cluster.Match(cluster) && Service.Match(service);
    }
};

IOutputStream& operator<<(IOutputStream& os, const TShardSelector& selector);

} // namespace NSolomon::NDataProxy
