#include <solomon/services/dataproxy/lib/selectors/shard_selector.h>

#include <library/cpp/testing/gtest/gtest.h>

#include <util/string/cast.h>

using namespace NSolomon;
using namespace NDataProxy;

TEST(TShardSelectorTest, ToString) {
    TShardSelector s1{
            "solomon",
            TSelector{"cluster", "production", false},
            TSelector{"service", "dataproxy", false},
    };
    ASSERT_EQ("{'project' == 'solomon', 'cluster' == 'production', 'service' == 'dataproxy'}", ToString(s1));

    TShardSelector s2{
            "solomon",
            TSelector{"cluster", AnyMatcher(), false},
            TSelector{"service", RegexMatcher("[a-z]+"), false}
    };
    ASSERT_EQ("{'project' == 'solomon', 'cluster' = '*', 'service' =~ '[a-z]+'}", ToString(s2));
}

TEST(TShardSelectorTest, FromPcs) {
    auto s = TShardSelector::FromPcs("solomon", "prestable", "dataproxy");
    EXPECT_EQ(s.Project, "solomon");

    EXPECT_EQ(s.Cluster.Key(), "cluster");
    EXPECT_EQ(s.Cluster.Pattern(), "prestable");
    EXPECT_TRUE(s.Cluster.IsExact());

    EXPECT_EQ(s.Service.Key(), "service");
    EXPECT_EQ(s.Service.Pattern(), "dataproxy");
    EXPECT_TRUE(s.Service.IsExact());

    EXPECT_TRUE(s.IsExact());
}

TEST(TShardSelectorTest, FromSelectorsExact) {
    auto s = TShardSelector::FromSelectors("solomon", ParseSelectors("{service=fetcher, cluster=production}"));
    EXPECT_EQ(s.Project, "solomon");

    EXPECT_EQ(s.Service.Key(), "service");
    EXPECT_EQ(s.Service.Pattern(), "fetcher");
    EXPECT_TRUE(s.Service.IsExact());

    EXPECT_EQ(s.Cluster.Key(), "cluster");
    EXPECT_EQ(s.Cluster.Pattern(), "production");
    EXPECT_TRUE(s.Cluster.IsExact());

    EXPECT_TRUE(s.IsExact());

    EXPECT_TRUE(s.Match("production", "fetcher"));
    EXPECT_FALSE(s.Match("prestable", "fetcher"));
    EXPECT_FALSE(s.Match("production", "coremon"));
}

TEST(TShardSelectorTest, FromSelectorsGlob) {
    auto s = TShardSelector::FromSelectors("solomon", ParseSelectors("{service=yasm, cluster=group_*}"));
    EXPECT_EQ(s.Project, "solomon");

    EXPECT_EQ(s.Service.Key(), "service");
    EXPECT_EQ(s.Service.Pattern(), "yasm");
    EXPECT_TRUE(s.Service.IsExact());

    EXPECT_EQ(s.Cluster.Key(), "cluster");
    EXPECT_EQ(s.Cluster.Pattern(), "group_*");
    EXPECT_FALSE(s.Cluster.IsExact());

    EXPECT_FALSE(s.IsExact());

    EXPECT_TRUE(s.Match("group_0", "yasm"));
    EXPECT_TRUE(s.Match("group_1", "yasm"));
    EXPECT_FALSE(s.Match("host_0", "yasm"));
    EXPECT_FALSE(s.Match("group_2", "fetcher"));
}

TEST(TShardSelectorTest, FromSelectorsAny) {
    {
        auto s = TShardSelector::FromSelectors("solomon", ParseSelectors("{service=yasm}"));
        EXPECT_EQ(s.Project, "solomon");

        EXPECT_EQ(s.Service.Key(), "service");
        EXPECT_EQ(s.Service.Pattern(), "yasm");
        EXPECT_TRUE(s.Service.IsExact());

        EXPECT_EQ(s.Cluster.Key(), "cluster");
        EXPECT_EQ(s.Cluster.Pattern(), "*");
        EXPECT_FALSE(s.Cluster.IsExact());

        EXPECT_FALSE(s.IsExact());

        EXPECT_TRUE(s.Match("group_0", "yasm"));
        EXPECT_TRUE(s.Match("group_1", "yasm"));
        EXPECT_TRUE(s.Match("host_0", "yasm"));
        EXPECT_FALSE(s.Match("group_2", "fetcher"));
    }
    {
        auto s = TShardSelector::FromSelectors("solomon", ParseSelectors("{cluster=group_*}"));
        EXPECT_EQ(s.Project, "solomon");

        EXPECT_EQ(s.Service.Key(), "service");
        EXPECT_EQ(s.Service.Pattern(), "*");
        EXPECT_FALSE(s.Service.IsExact());

        EXPECT_EQ(s.Cluster.Key(), "cluster");
        EXPECT_EQ(s.Cluster.Pattern(), "group_*");
        EXPECT_FALSE(s.Cluster.IsExact());

        EXPECT_FALSE(s.IsExact());

        EXPECT_TRUE(s.Match("group_0", "yasm"));
        EXPECT_TRUE(s.Match("group_1", "yasm"));
        EXPECT_FALSE(s.Match("host_0", "yasm"));
        EXPECT_TRUE(s.Match("group_2", "fetcher"));
    }
    {
        auto s = TShardSelector::FromSelectors("solomon", TSelectors{});
        EXPECT_EQ(s.Project, "solomon");

        EXPECT_EQ(s.Service.Key(), "service");
        EXPECT_EQ(s.Service.Pattern(), "*");
        EXPECT_FALSE(s.Service.IsExact());

        EXPECT_EQ(s.Cluster.Key(), "cluster");
        EXPECT_EQ(s.Cluster.Pattern(), "*");
        EXPECT_FALSE(s.Cluster.IsExact());

        EXPECT_FALSE(s.IsExact());

        EXPECT_TRUE(s.Match("group_0", "yasm"));
        EXPECT_TRUE(s.Match("group_1", "yasm"));
        EXPECT_TRUE(s.Match("host_0", "yasm"));
        EXPECT_TRUE(s.Match("group_2", "fetcher"));
        EXPECT_TRUE(s.Match("production", "coremon"));
    }
}

TEST(TShardSelectorTest, FromLabelsExact) {
    auto s = TShardSelector::FromLabels("solomon", TLabels<TString>{{"cluster", "production"}, {"service", "fetcher"}});
    EXPECT_EQ(s.Project, "solomon");

    EXPECT_EQ(s.Service.Key(), "service");
    EXPECT_EQ(s.Service.Pattern(), "fetcher");
    EXPECT_TRUE(s.Service.IsExact());

    EXPECT_EQ(s.Cluster.Key(), "cluster");
    EXPECT_EQ(s.Cluster.Pattern(), "production");
    EXPECT_TRUE(s.Cluster.IsExact());

    EXPECT_TRUE(s.IsExact());

    EXPECT_TRUE(s.Match("production", "fetcher"));
    EXPECT_FALSE(s.Match("prestable", "fetcher"));
    EXPECT_FALSE(s.Match("production", "coremon"));
}

TEST(TShardSelectorTest, FromLabelsAny) {
    ASSERT_THROW_MESSAGE_HAS_SUBSTR({
        TShardSelector::FromLabels("solomon", TLabels<TString>{{"cluster", "production"}});
    }, yexception, "missed or empty 'service' label");

    ASSERT_THROW_MESSAGE_HAS_SUBSTR({
        TShardSelector::FromLabels("solomon", TLabels<TString>{{"service", "fetcher"}});
    }, yexception, "missed or empty 'cluster' label");
}

TEST(TShardSelectorTest, FromInternedLabelsExact) {
    NStringPool::TStringPoolBuilder sb;
    ui32 cluster = sb.Put("cluster");
    ui32 production = sb.Put("production");
    ui32 service = sb.Put("service");
    ui32 fetcher = sb.Put("fetcher");

    auto strings = sb.Build();

    auto s = TShardSelector::FromLabels("solomon", strings, {{cluster, production}, {service, fetcher}});
    EXPECT_EQ(s.Project, "solomon");

    EXPECT_EQ(s.Service.Key(), "service");
    EXPECT_EQ(s.Service.Pattern(), "fetcher");
    EXPECT_TRUE(s.Service.IsExact());

    EXPECT_EQ(s.Cluster.Key(), "cluster");
    EXPECT_EQ(s.Cluster.Pattern(), "production");
    EXPECT_TRUE(s.Cluster.IsExact());

    EXPECT_TRUE(s.IsExact());

    EXPECT_TRUE(s.Match("production", "fetcher"));
    EXPECT_FALSE(s.Match("prestable", "fetcher"));
    EXPECT_FALSE(s.Match("production", "coremon"));
}

TEST(TShardSelectorTest, FromInternedLabelsAny) {
    NStringPool::TStringPoolBuilder sb;
    ui32 cluster = sb.Put("cluster");
    ui32 production = sb.Put("production");
    ui32 service = sb.Put("service");
    ui32 fetcher = sb.Put("fetcher");

    auto strings = sb.Build();

    ASSERT_THROW_MESSAGE_HAS_SUBSTR({
        TShardSelector::FromLabels("solomon", strings, {{cluster, production}});
    }, yexception, "missed or empty 'service' label");

    ASSERT_THROW_MESSAGE_HAS_SUBSTR({
        TShardSelector::FromLabels("solomon", strings, {{service, fetcher}});
    }, yexception, "missed or empty 'cluster' label");
}
