#pragma once

#include <util/generic/string.h>
#include <util/digest/multi.h>

namespace NSolomon::NDataProxy {

/**
 * Shard key in some project.
 */
struct TShardSubKey {
    TString Cluster;
    TString Service;

    TShardSubKey() noexcept = default;

    TShardSubKey(TString cluster, TString service) noexcept
        : Cluster{std::move(cluster)}
        , Service{std::move(service)}
    {
    }

    bool operator==(const TShardSubKey& rhs) const noexcept = default;
    bool operator!=(const TShardSubKey& rhs) const noexcept = default;

    operator size_t() const noexcept {
        return MultiHash(Cluster, Service);
    }
};

/**
 * Global shard key.
 */
struct TShardKey {
    TString Project;
    TShardSubKey SubKey;

    TShardKey() noexcept = default;

    TShardKey(TString project, TShardSubKey subKey) noexcept
        : Project{std::move(project)}
        , SubKey{std::move(subKey)}
    {
    }

    TShardKey(TString project, TString cluster, TString service) noexcept
        : Project{std::move(project)}
        , SubKey{std::move(cluster), std::move(service)}
    {
    }

    bool operator==(const TShardKey& rhs) const noexcept = default;
    bool operator!=(const TShardKey& rhs) const noexcept = default;

    operator size_t() const noexcept {
        return MultiHash(Project, SubKey);
    }
};

IOutputStream& operator<<(IOutputStream& os, const TShardSubKey& subKey);
IOutputStream& operator<<(IOutputStream& os, const TShardKey& key);

} // namespace NSolomon::NDataProxy

template <>
struct std::hash<NSolomon::NDataProxy::TShardSubKey> {
    size_t operator()(const NSolomon::NDataProxy::TShardSubKey& subKey) const noexcept {
        return static_cast<size_t>(subKey);
    }
};

template <>
struct std::hash<NSolomon::NDataProxy::TShardKey> {
    size_t operator()(const NSolomon::NDataProxy::TShardKey& key) const noexcept {
        return static_cast<size_t>(key);
    }
};
