#include "shards_map.h"

namespace NSolomon::NDataProxy {

bool TShardsMap::Update(const TShardInfoPtr& shard) {
    bool isUpdated = false;

    const TShardKey& key = shard->Key;
    TProjectShards& projectShards = Shards_[key.Project];

    if (auto it = ShardsById_.find(shard->Id); it != ShardsById_.end()) {
        if ((*shard) != (*it->second)) {
            isUpdated = true;
        }

        const TShardSubKey& prevSubKey = it->second->Key.SubKey;
        if (prevSubKey != key.SubKey) {
            // if cluster or service was changed, then
            // remove previous {shardKey -> shard} association
            projectShards.erase(prevSubKey);
        }
        it->second = shard;
        projectShards[key.SubKey] = shard;
    } else {
        ShardsById_.emplace(shard->Id, shard);
        projectShards.emplace(key.SubKey, shard);
    }

    return isUpdated;
}

TShardInfoPtr TShardsMap::Remove(const TShardId& id) {
    if (auto it = ShardsById_.find(id); it != ShardsById_.end()) {
        auto shardInfo = it->second;

        const TShardKey& key = it->second->Key;

        TProjectShards& projectShards = Shards_[key.Project];
        projectShards.erase(key.SubKey);

        if (projectShards.empty()) {
            Shards_.erase(key.Project);
        }

        ShardsById_.erase(it);

        return shardInfo;
    }

    return nullptr;
}

std::optional<TShardLocation> TShardsMap::FindExact(const TShardSelector& selector) const {
    if (auto shard = FindExactInfo(selector)) {
        return TShardLocation{shard->Id, shard->Address};
    }
    return std::nullopt;
}

TShardInfoPtr TShardsMap::FindExactInfo(const TShardSelector& selector) const {
    Y_VERIFY_DEBUG(selector.IsExact());

    auto projectId = Shards_.find(selector.Project);
    if (projectId == Shards_.end()) {
        return nullptr;
    }

    TShardSubKey subKey{
            TString{selector.Cluster.Pattern()},
            TString{selector.Service.Pattern()}};

    const TProjectShards& projectShards = projectId->second;
    auto shardIt = projectShards.find(subKey);
    if (shardIt == projectShards.end()) {
        return nullptr;
    }

    return shardIt->second;
}

std::vector<TShardLocation> TShardsMap::Find(const TShardSelector& selector) const {
    auto projectId = Shards_.find(selector.Project);
    if (projectId == Shards_.end()) {
        return {};
    }

    const TProjectShards& projectShards = projectId->second;

    std::vector<TShardLocation> locations;
    for (auto& [subKey, shard]: projectShards) {
        if (selector.Match(subKey.Cluster, subKey.Service)) {
            locations.emplace_back(shard->Id, shard->Address);
        }
    }
    return locations;
}

std::vector<TShardInfoPtr> TShardsMap::FindInfo(const TShardSelector& selector) const {
    auto projectId = Shards_.find(selector.Project);
    if (projectId == Shards_.end()) {
        return {};
    }

    const TProjectShards& projectShards = projectId->second;

    std::vector<TShardInfoPtr> result;
    for (auto& [subKey, shard]: projectShards) {
        if (selector.Match(subKey.Cluster, subKey.Service)) {
            result.emplace_back(shard);
        }
    }
    return result;
}

} // namespace NSolomon::NDataProxy
