#include <solomon/services/dataproxy/lib/shard/shards_map.h>

#include <library/cpp/testing/gtest/gtest.h>

using namespace NSolomon;
using namespace NDataProxy;

namespace {

TShardInfoPtr Shard(TShardId id, TString project, TString cluster, TString service, TString address) {
    TShardKey key{std::move(project), std::move(cluster), std::move(service)};
    return std::make_shared<TShardInfo>(id, std::move(address), std::move(key), true);
}

} // namespace

TEST(TShardsMapTest, Empty) {
    TShardsMap map;
    EXPECT_EQ(map.Size(), 0u);

    auto selector = TShardSelector::FromPcs("solomon", "production", "dataproxy");

    auto location = map.FindExact(selector);
    EXPECT_FALSE(location.has_value());

    auto shard = map.FindExactInfo(selector);
    EXPECT_FALSE(location);

    auto locations = map.Find(selector);
    EXPECT_TRUE(locations.empty());

    auto shards = map.FindInfo(selector);
    EXPECT_TRUE(shards.empty());
}

TEST(TShardsMapTest, Update) {
    TShardsMap map;
    EXPECT_EQ(map.Size(), 0u);

    map.Update({ Shard(42u, "solomon", "production", "dataproxy", "host-aaa") });
    EXPECT_EQ(map.Size(), 1u);

    auto selector = TShardSelector::FromPcs("solomon", "production", "dataproxy");

    auto location = map.FindExact(selector);
    ASSERT_TRUE(location.has_value());
    EXPECT_EQ(location->Id, 42u);
    EXPECT_EQ(location->Address, "host-aaa");

    auto shard = map.FindExactInfo(selector);
    ASSERT_TRUE(shard);
    EXPECT_EQ(shard->Id, 42u);
    EXPECT_EQ(shard->Address, "host-aaa");
    EXPECT_EQ(shard->Key, TShardKey("solomon", "production", "dataproxy"));
    EXPECT_TRUE(shard->Ready);

    auto locations = map.Find(selector);
    ASSERT_EQ(locations.size(), 1u);
    EXPECT_EQ(locations[0].Id, 42u);
    EXPECT_EQ(locations[0].Address, "host-aaa");

    auto shards = map.FindInfo(selector);
    ASSERT_EQ(shards.size(), 1u);
    EXPECT_EQ(shards[0]->Id, 42u);
    EXPECT_EQ(shards[0]->Address, "host-aaa");
    EXPECT_EQ(shards[0]->Key, TShardKey("solomon", "production", "dataproxy"));
    EXPECT_TRUE(shards[0]->Ready);
}

TEST(TShardsMapTest, Replace) {
    TShardsMap map;
    EXPECT_EQ(map.Size(), 0u);

    map.Update({ Shard(42u, "solomon", "production", "dataproxy", "host-aaa") });
    EXPECT_EQ(map.Size(), 1u);

    map.Update({ Shard(42u, "solomon", "production", "dataproxy", "host-bbb") });
    EXPECT_EQ(map.Size(), 1u);

    auto selector = TShardSelector::FromPcs("solomon", "production", "dataproxy");

    auto shard = map.FindExact(selector);
    ASSERT_TRUE(shard.has_value());
    EXPECT_EQ(shard->Id, 42u);
    EXPECT_EQ(shard->Address, "host-bbb");

    auto shards = map.Find(selector);
    ASSERT_EQ(shards.size(), 1u);
    EXPECT_EQ(shards[0].Id, 42u);
    EXPECT_EQ(shards[0].Address, "host-bbb");
}

TEST(TShardsMapTest, ReplaceWithDifferentKey) {
    TShardsMap map;
    EXPECT_EQ(map.Size(), 0u);

    map.Update({ Shard(42u, "solomon", "production", "dataproxy", "host-aaa") });
    EXPECT_EQ(map.Size(), 1u);

    map.Update({ Shard(42u, "solomon", "testing", "dataproxy", "host-bbb") });
    EXPECT_EQ(map.Size(), 1u);

    {
        // there is no shard with replaced shard key
        auto selector = TShardSelector::FromPcs("solomon", "production", "dataproxy");

        auto shard = map.FindExact(selector);
        EXPECT_FALSE(shard.has_value());

        auto shards = map.Find(selector);
        EXPECT_TRUE(shards.empty());
    }

    {
        // and exactly one shard with the new shard key
        auto selector = TShardSelector::FromPcs("solomon", "testing", "dataproxy");

        auto shard = map.FindExact(selector);
        ASSERT_TRUE(shard.has_value());
        EXPECT_EQ(shard->Id, 42u);
        EXPECT_EQ(shard->Address, "host-bbb");

        auto shards = map.Find(selector);
        ASSERT_EQ(shards.size(), 1u);
        EXPECT_EQ(shards[0].Id, 42u);
        EXPECT_EQ(shards[0].Address, "host-bbb");
    }
}

TEST(TShardsMapTest, Remove) {
    TShardsMap map;
    EXPECT_EQ(map.Size(), 0u);

    map.Update(Shard(42u, "solomon", "production", "dataproxy", "host-aaa"));
    map.Update(Shard(43u, "solomon", "production", "coremon", "host-bbb"));
    map.Update(Shard(44u, "solomon", "production", "stockpile", "host-aaa"));
    map.Update(Shard(45u, "solomon", "production", "gateway", "host-ccc"));
    map.Update(Shard(46u, "yasm", "testing", "gateway", "host-xxx"));

    EXPECT_EQ(map.Size(), 5u);

    {
        auto selector = TShardSelector::FromPcs("solomon", "production", "coremon");

        auto shard1 = map.FindExact(selector);
        ASSERT_TRUE(shard1.has_value());
        EXPECT_EQ(shard1->Id, 43u);
        EXPECT_EQ(shard1->Address, "host-bbb");

        map.Remove({ 43u });
        EXPECT_EQ(map.Size(), 4u);

        auto shard2 = map.FindExact(selector);
        ASSERT_FALSE(shard2.has_value());
    }

    {
        auto selector = TShardSelector::FromPcs("yasm", "testing", "gateway");

        auto shard1 = map.FindExact(selector);
        ASSERT_TRUE(shard1.has_value());
        EXPECT_EQ(shard1->Id, 46u);
        EXPECT_EQ(shard1->Address, "host-xxx");

        map.Remove({ 46u });
        EXPECT_EQ(map.Size(), 3u);

        auto shard2 = map.FindExact(selector);
        ASSERT_FALSE(shard2.has_value());
    }

    map.Remove(42u);
    map.Remove(44u);
    map.Remove(45u);
    EXPECT_EQ(map.Size(), 0u);
}

TEST(TShardsMapTest, UpdateReturnValues) {
    TShardsMap map;
    EXPECT_EQ(map.Size(), 0u);

    // shards were only added, no _updates_ were performed
    EXPECT_FALSE(map.Update(Shard(42u, "solomon", "production", "dataproxy", "host-aaa")));
    EXPECT_EQ(map.Size(), 1u);

    EXPECT_TRUE(map.Update(Shard(42u, "solomon", "production", "dataproxy", "host-bbb")));
    EXPECT_EQ(map.Size(), 1u);

    EXPECT_TRUE(map.Update(Shard(42u, "solomon", "production", "dataproxy", "host-ccc")));
    EXPECT_EQ(map.Size(), 1u);

    // nothing changes
    EXPECT_FALSE(map.Update(Shard(42u, "solomon", "production", "dataproxy", "host-ccc")));

    auto removeInfo = map.Remove(42u);
    EXPECT_EQ(map.Size(), 0u);

    {
        EXPECT_TRUE(removeInfo);

        EXPECT_EQ(removeInfo->Address, "host-ccc");
        EXPECT_EQ(removeInfo->Id, 42u);
    }
}
