#include <solomon/services/dataproxy/lib/stockpile/rpc.h>

#include <solomon/libs/cpp/host_resolver/host_resolver.h>
#include <solomon/libs/cpp/stockpile_codec/metric_archive.h>
#include <solomon/libs/cpp/ts_codec/columns.h>
#include <solomon/libs/cpp/ts_codec/double_ts_codec.h>

#include <library/cpp/getopt/last_getopt.h>
#include <library/cpp/monlib/metrics/metric_registry.h>

#include <util/stream/null.h>
#include <util/stream/format.h>

using namespace NSolomon;
using namespace NDataProxy;
using namespace yandex::solomon::config;
using namespace yandex::solomon::model;
using namespace yandex::solomon::stockpile;

IOutputStream* LoggerPtr = &Cnull;
#define LOGGER (*LoggerPtr)

std::map<TStringBuf, TStringBuf> Cluster2Group = {
        {"test", "solomon_test_data_storage"},
        {"pre", "solomon_pre_stockpile"},
        {"prod_sas", "solomon_prod_stockpile_sas"},
        {"prod_vla", "solomon_prod_stockpile_vla"},
};

TStringBuf FindGroup(TStringBuf cluster) {
    if (auto it = Cluster2Group.find(cluster); it == Cluster2Group.end()) {
        throw TBadArgumentException() << "unknown cluster: " << cluster;
    } else {
        return it->second;
    }
}

std::pair<ui32, ui64> ParseMetricId(TStringBuf str) {
    TStringBuf shardIdStr, localIdStr;
    Y_ENSURE_EX(str.TrySplit('/', shardIdStr, localIdStr), TBadArgumentException()
            << "invalid metricId format (must be <shardId>/<localId>), but got: " << str);

    ui32 shardId;
    Y_ENSURE_EX(TryFromString(shardIdStr, shardId), TBadArgumentException()
            << "cannot parse shardId from: " << shardIdStr);

    ui64 localId;
    Y_ENSURE_EX(TryFromString(localIdStr, localId), TBadArgumentException()
            << "cannot parse localId from: " << localIdStr);

    return {shardId, localId};
}

rpc::TGrpcClientConfig RpcConfig(TStringBuf group, ui16 port) {
    TAddressSet addresses;
    TString groupWithScheme = TString{"conductor_group://"} + group;
    ConductorGroupResolver()->Resolve(groupWithScheme, &addresses);

    rpc::TGrpcClientConfig config;
    for (const auto& address: addresses) {
        config.add_addresses(TStringBuilder{} << address << ':' << port);
    }

    auto setTimeoutSeconds = [](auto* timeout, int value) {
        timeout->set_value(value);
        timeout->set_unit(TimeUnit::SECONDS);
    };

    setTimeoutSeconds(config.mutable_readtimeout(), 20);
    setTimeoutSeconds(config.mutable_connecttimeout(), 5);
    return config;
}

std::unordered_map<ui32, TStringBuf> LoadShardLocations(IStockpileClusterRpcPtr& stockpileRpc) {
    TServerStatusRequest serverStatusRequest;
    serverStatusRequest.set_deadline(TDuration::Seconds(20).ToDeadLine().MilliSeconds());

    std::vector<TAsyncStockpileStatusResponse> statusFutures;
    const auto& addresses = stockpileRpc->Addresses();
    for (TStringBuf address: addresses) {
        auto* nodeRpc = stockpileRpc->Get(address);
        LOGGER << "send ServerStatus() to " << address << '\n';
        statusFutures.push_back(nodeRpc->ServerStatus(serverStatusRequest));
    }

    LOGGER << "await status responses ";
    for (auto done = NThreading::WaitExceptionOrAll(statusFutures); !done.Wait(TDuration::Seconds(1)); ) {
        LOGGER << '.';
    }
    LOGGER << Endl;

    std::unordered_map<ui32, TStringBuf> shardLocations;
    for (size_t i = 0; i < statusFutures.size(); ++i) {
        const auto& responseOrError = statusFutures[i].GetValueSync();
        if (responseOrError.Fail()) {
            Cerr << "cannot get status response from " << addresses[i] << ' ' << responseOrError.Error() << Endl;
            continue;
        }

        const TServerStatusResponse& response = responseOrError.Value();
        if (response.GetStatus() != EStockpileStatusCode::OK) {
            TStockpileError error{grpc::StatusCode::OK, response.GetStatus(), response.GetStatusMessage()};
            Cerr << "cannot get status response from " << addresses[i] << ' ' << error << Endl;
            continue;
        }

        for (const auto& shardStatus: response.GetShardStatus()) {
            shardLocations[shardStatus.GetShardId()] = addresses[i];
        }
    }

    return shardLocations;
}

template <typename T>
const T& ExpectOk(const TErrorOr<T, TStockpileError>& responseOrError) {
    Y_ENSURE(!responseOrError.Fail(), "cannot read data " << responseOrError.Error());

    const T& response = responseOrError.Value();
    EStockpileStatusCode status = response.GetStatus();
    Y_ENSURE(status == EStockpileStatusCode::OK,
            "cannot read data " << TStockpileError(grpc::StatusCode::OK, status, response.GetStatusMessage()));

    return response;
}

void ReadOne(IStockpileRpc* nodeRpc, const TReadRequest& readRequest) {
    TAsyncReadResponse future = nodeRpc->ReadOne(readRequest);
    const TReadResponse& response = ExpectOk(future.GetValueSync());
    LOGGER << "response size: " << HumanReadableSize(response.ByteSizeLong(), SF_BYTES) << " bytes\n";

    NTs::TColumnSet columns{static_cast<NTs::TColumnsMask>(response.GetColumnMask())};
    Cout << "type: " << MetricType_Name(response.GetType()) << '\n';
    Cout << "columns: " << columns << '\n';
    Cout << "points count: " << response.PointsSize() << '\n';

    for (const TPoint& p: response.GetPoints()) {
        Cout << "    " << TInstant::MilliSeconds(p.GetTimestampsMillis()) << ' ' << p.GetDoubleValue() << '\n';
    }
}

void ReadCompressedOne(IStockpileRpc* nodeRpc, const TReadRequest& readRequest) {
    TAsyncCompressedReadResponse future = nodeRpc->ReadCompressedOne(readRequest);
    const TCompressedReadResponse& response = ExpectOk(future.GetValueSync());
    LOGGER << "response size: " << HumanReadableSize(response.ByteSizeLong(), SF_BYTES) << " bytes\n";

    i32 binaryVersion = response.GetBinaryVersion();
    Y_ENSURE((binaryVersion == static_cast<i32>(NStockpile::EFormat::HISTOGRAM_DENOM_37)) ||
            (binaryVersion == static_cast<i32>(NStockpile::EFormat::IDEMPOTENT_WRITE_38)),
            "unsupported stockpile format: " << binaryVersion);

    Cout << "type: " << MetricType_Name(response.GetType()) << '\n';
    Cout << "format: " << binaryVersion << '\n';

    NStockpile::TMetricArchiveCodec codec{static_cast<NStockpile::EFormat>(binaryVersion)};
    for (const auto& chunk: response.GetChunks()) {
        NStockpile::TCodecInput input{chunk.content()};
        NStockpile::TMetricArchive archive = codec.Decode(&input);

        Cout << "columns: " << archive.Columns() << '\n';
        Cout << "points count: " << archive.PointCount() << '\n';

        NTs::TDoubleTsDecoder tsDecoder{archive.Columns(), archive.Data()};
        for (NTs::TDoublePoint point; tsDecoder.NextPoint(&point); ) {
            Cout << "    " << point.Time << ' ' << point.ValueDivided() << '\n';
        }
    }
}

void Main(const NLastGetopt::TOptsParseResult& opts) {
    if (opts.Has("verbose")) {
        LoggerPtr = &Cerr;
    }

    auto freeArgs = opts.GetFreeArgs();
    auto metricId = ParseMetricId(freeArgs[0]);

    ui16 port = opts.Get<ui16>("port");
    TStringBuf group = FindGroup(opts.Get<TStringBuf>("cluster"));
    bool uncompressed = opts.Has("uncompressed");

    NMonitoring::TMetricRegistry registry;
    rpc::TGrpcClientConfig rpcConfig = RpcConfig(group, port);
    auto stockpileRpc = CreateStockpileRpc(rpcConfig, registry);

    auto shardLocations = LoadShardLocations(stockpileRpc);
    TStringBuf address = shardLocations[metricId.first];
    Y_ENSURE(address, "cannot find shard " << metricId.first << " location");

    TReadRequest readRequest;
    readRequest.SetBinaryVersion(static_cast<i32>(NStockpile::EFormat::IDEMPOTENT_WRITE_38));
    {
        auto* id = readRequest.mutable_metric_id();
        id->set_shard_id(metricId.first);
        id->set_local_id(metricId.second);
    }

    if (uncompressed) {
        LOGGER << "send ReadOne(" << readRequest.ShortDebugString() << ") to " << address << '\n';
        ReadOne(stockpileRpc->Get(address), readRequest);
    } else {
        LOGGER << "send ReadCompressedOne(" << readRequest.ShortDebugString() << ") to " << address << '\n';
        ReadCompressedOne(stockpileRpc->Get(address), readRequest);
    }
}

int main(int argc, const char* argv[]) {
    NLastGetopt::TOpts opts;
    opts.AddLongOption("cluster", "Stockpile cluster, one of {test, pre, prod_sas, prod_vla}")
            .RequiredArgument("STR")
            .Required();
    opts.AddLongOption("port", "Stockpile port")
            .RequiredArgument("NUM")
            .DefaultValue("5700")
            .Optional();
    opts.AddLongOption("uncompressed", "Read data in uncompressed form")
            .NoArgument()
            .Optional();
    opts.AddLongOption("verbose", "Print more information about network communications")
            .NoArgument()
            .Optional();
    opts.SetFreeArgsNum(1);
    opts.SetFreeArgTitle(0, "metricId", "metric id in format <shardId>/<localId>");

    try {
        NLastGetopt::TOptsParseResult res{&opts, argc, argv};
        Main(res);
        return 0;
    } catch (const TBadArgumentException& e) {
        Cerr << "Bad argument." << '\n' << e.AsStrBuf() << Endl;
    } catch (const NLastGetopt::TUsageException&) {
        opts.PrintUsage("stockpile-bin");
    } catch (...) {
        Cerr << "unhandled exception: " << CurrentExceptionMessage() << Endl;
    }
    return 1;
}
