#include "cluster.h"
#include "events.h"
#include "watcher.h"
#include "shard_actor.h"

#include <solomon/services/dataproxy/lib/initialization/events.h>

#include <solomon/libs/cpp/logging/logging.h>
#include <solomon/libs/cpp/selfmon/selfmon.h>

#include <library/cpp/actors/core/actor_bootstrapped.h>
#include <library/cpp/actors/core/event.h>
#include <library/cpp/actors/core/hfunc.h>

using namespace NActors;
using namespace yandex::solomon::stockpile;

namespace NSolomon::NDataProxy {
namespace {

class TStockpileCluster: public TActorBootstrapped<TStockpileCluster> {
public:
    TStockpileCluster(std::vector<TString> addresses, IStockpileClusterRpcPtr rpc, TStockpileConfig config)
        : Addresses_{std::move(addresses)}
        , Rpc_{std::move(rpc)}
        , Config_{std::move(config)}
    {
    }

    void Bootstrap() {
        WatcherId_ = Register(StockpileClusterWatcher(Rpc_, Addresses_, TDuration::Seconds(3)).release());
        Send(WatcherId_, new TStockpileWatcherEvents::TSubscribe);
        Become(&TThis::Normal);
    }

    STFUNC(Normal) {
        switch (ev->GetTypeRewrite()) {
            hFunc(TStockpileWatcherEvents::TUpdate, OnStateChange);
            hFunc(TEvents::TEvPoison, OnPoison);
            hFunc(TStockpileEvents::TReadOneReq, OnReadOne);
            hFunc(TStockpileEvents::TReadManyReq, OnReadMany);
            HFunc(TInitializationEvents::TSubscribe, OnInitializationSubscribe);
            HFunc(NSelfMon::TEvPageDataReq, OnSelfMon);
        }
    }

    STATEFN(Dying) {
        switch (ev->GetTypeRewrite()) {
            hFunc(TEvents::TEvPoisonTaken, OnPoisonTaken);
        }
    }

    static constexpr char ActorName[] = "StockpileCluster";

private:
    void OnStateChange(TStockpileWatcherEvents::TUpdate::TPtr& ev) {
        for (auto& shard: ev->Get()->Shards) {
            if (auto it = ShardIdToActorId_.find(shard.Id); it != ShardIdToActorId_.end()) {
                auto updateShard = MakeHolder<TStockpileWatcherEvents::TUpdate>();
                updateShard->Shards.push_back(shard);
                Send(it->second, std::move(updateShard));
            }
            ShardsMap_[shard.Id] = std::move(shard);
        }
    }

    TActorId GetOrCreateShardActor(TStockpileShardId shardId, const TStockpileShardInfo& shardInfo) {
        auto& actorId = ShardIdToActorId_[shardId];
        if (!actorId) {
            auto maxInflight = Config_.GetRequestInflightPerShard();
            actorId = Register(StockpileShardActor(Rpc_, shardInfo, maxInflight).release());
            StockpileShardActors_.push_back(actorId);
        }
        return actorId;
    }

    void OnReadOne(TStockpileEvents::TReadOneReq::TPtr& ev) {
        TReadRequest& req = ev->Get()->Message;
        HandleRequest(ev, req.metric_id().shard_id());
    }

    void OnReadMany(TStockpileEvents::TReadManyReq::TPtr& ev) {
        TReadManyRequest& req = ev->Get()->Message;

        HandleRequest(ev, req.GetShardId());
    }

    template <typename TReqEventPtr>
    void HandleRequest(TReqEventPtr& ev, TStockpileShardId shardId) {
        const auto& req = ev->Get()->Message;

        if (auto it = ShardsMap_.find(shardId); it != ShardsMap_.end()) {
            MON_DEBUG(StockpileClient, req.GetTypeName() << " {" << req.ShortDebugString() << "} on " << it->second.Location);
            auto requesterId = GetOrCreateShardActor(shardId, it->second);
            TActivationContext::Send(ev->Forward(requesterId));
            return;
        }

        MON_WARN(StockpileClient, "cannot find shard by id " << shardId << ", request: " << req.GetTypeName() << " { " << req.ShortDebugString() << " }");
        auto event = std::make_unique<TStockpileEvents::TError>();
        event->RpcCode = grpc::StatusCode::UNKNOWN;
        event->StockpileCode = EStockpileStatusCode::SHARD_NOT_READY;
        event->Message = TStringBuilder{} << "shard " << shardId << " not found";
        this->Send(ev->Sender, event.release(), 0, ev->Cookie, std::move(ev->TraceId));
    }

    void OnPoison(TEvents::TEvPoison::TPtr& ev) {
        PoisonerId_ = ev->Sender;
        for (auto& shardActorId: StockpileShardActors_) {
            Send(shardActorId, new TEvents::TEvPoison);
        }
        Become(&TThis::Dying);
    }

    void OnPoisonTaken(TEvents::TEvPoisonTaken::TPtr& ev) {
        Send(PoisonerId_, ev->Release().Release());
        PassAway();
    }

    void OnSelfMon(const NSelfMon::TEvPageDataReq::TPtr& ev, const TActorContext& ctx) {
        ctx.Send(ev->Forward(WatcherId_));
    }

    void OnInitializationSubscribe(TInitializationEvents::TSubscribe::TPtr& ev, const TActorContext& ctx) {
        ctx.Send(ev->Forward(WatcherId_));
    }

private:
    const std::vector<TString> Addresses_;
    IStockpileClusterRpcPtr Rpc_;
    TActorId WatcherId_;
    TStockpileShardsMap ShardsMap_;
    TActorId PoisonerId_;
    std::unordered_map<TStockpileShardId, TActorId> ShardIdToActorId_;
    TVector<TActorId> StockpileShardActors_;
    TStockpileConfig Config_;
};

} // namespace

std::unique_ptr<IActor> StockpileCluster(std::vector<TString> addresses, IStockpileClusterRpcPtr rpc, TStockpileConfig config) {
    return std::make_unique<TStockpileCluster>(std::move(addresses), std::move(rpc), std::move(config));
}

} // namespace NSolomon::NDataProxy
