#include "rpc.h"

#include <solomon/libs/cpp/config/units.h>
#include <solomon/libs/cpp/grpc/status/code.h>

#include <solomon/protos/stockpile/stockpile_service.grpc.pb.h>

#include <library/cpp/grpc/client/grpc_client_low.h>
#include <library/cpp/monlib/metrics/metric_registry.h>

namespace NSolomon::NDataProxy {
namespace {

using namespace NThreading;
using namespace NGrpc;
using namespace yandex::monitoring::stockpile;
using namespace yandex::solomon::stockpile;

using yandex::solomon::config::rpc::TGrpcClientConfig;
using yandex::solomon::config::DataUnit;
using yandex::solomon::config::TimeUnit;

/**
 * Actual Stockpile RPC implementation.
 */
class TStockpileRpcImpl final: public IStockpileRpc {
    using TStub = StockpileService::Stub;

public:
    TStockpileRpcImpl(
            NMonitoring::IMetricRegistry&,
            std::unique_ptr<TGrpcServiceConnection<StockpileService>> connection)
        : Connection_{std::move(connection)}
    {
    }

    TAsyncStockpileStatusResponse ServerStatus(const TServerStatusRequest& request) override {
        return SendRequest<TServerStatusRequest, TServerStatusResponse>(request, &TStub::AsyncServerStatus);
    }

    TAsyncReadResponse ReadOne(const TReadRequest& request) override {
        return SendRequest<TReadRequest, TReadResponse>(request, &TStub::AsyncReadOne);
    }

    TAsyncUncompressedReadManyResponse ReadUncompressedMany(const TReadManyRequest& request) override {
        return SendRequest<TReadManyRequest, TUncompressedReadManyResponse>(request, &TStub::AsyncReadUncompressedMany);
    }

    TAsyncCompressedReadResponse ReadCompressedOne(const TReadRequest& request) override {
        return SendRequest<TReadRequest, TCompressedReadResponse>(request, &TStub::AsyncReadCompressedOne);
    }

    TAsyncCompressedReadManyResponse ReadCompressedMany(const TReadManyRequest& request) override {
        return SendRequest<TReadManyRequest, TCompressedReadManyResponse>(request, &TStub::AsyncReadCompressedMany);
    }

    TAsyncReadMetricsMetaResponse ReadMetricsMeta(const ReadMetricsMetaRequest& request) override {
        return SendRequest<ReadMetricsMetaRequest, ReadMetricsMetaResponse>(request, &TStub::AsyncReadMetricsMeta);
    }

private:
    template <typename TReq, typename TResp, typename TProcessor = TSimpleRequestProcessor<StockpileService::Stub, TReq, TResp>>
    TFuture<TErrorOr<TResp, TStockpileError>> SendRequest(const TReq& req, typename TProcessor::TAsyncRequest asyncRequest) {
        using TResult = TErrorOr<TResp, TStockpileError>;

        auto promise = NThreading::NewPromise<TResult>();
        Connection_->Request<TReq, TResp>(
                req,
                [promise](TGrpcStatus&& status, TResp&& resp) mutable {
                    if (status.Ok()) {
                        if (resp.GetStatus() == EStockpileStatusCode::OK) {
                            promise.SetValue(std::move(resp));
                        } else {
                            promise.SetValue(TStockpileError{grpc::StatusCode::OK, resp.GetStatus(), resp.GetStatusMessage()});
                        }
                    } else {
                        auto rpcCode = static_cast<grpc::StatusCode>(status.GRpcStatusCode);
                        promise.SetValue(TStockpileError{rpcCode, EStockpileStatusCode::UNKNOWN, status.Msg});
                    }
                },
                asyncRequest,
                CreateCallMeta(TInstant::MilliSeconds(req.GetDeadline())));
        return promise.GetFuture();
    }

    TCallMeta CreateCallMeta(TInstant deadline) {
        TCallMeta meta;
        if (deadline) {
            meta.Timeout = deadline;
        } else {
            meta.Timeout = Connection_->Timeout();
        }
        return meta;
    }

private:
    std::unique_ptr<TGrpcServiceConnection<StockpileService>> Connection_;
};

IStockpileRpcPtr CreateStockpileClient(
        NMonitoring::IMetricRegistry& registry,
        std::unique_ptr<TGrpcServiceConnection<StockpileService>> connection)
{
    return std::make_unique<TStockpileRpcImpl>(registry, std::move(connection));
}

TGrpcClientConfig CreateConfig(const TString& address) {
    TGrpcClientConfig conf;
    conf.AddAddresses(address);
    conf.SetWorkerThreads(2);
    {
        auto* inMaxSize = conf.MutableMaxInboundMessageSize();
        inMaxSize->SetValue(16);
        inMaxSize->SetUnit(DataUnit::MEGABYTES);
    }
    {
        auto* outMaxSize = conf.MutableMaxOutboundMessageSize();
        outMaxSize->SetValue(1);
        outMaxSize->SetUnit(DataUnit::MEGABYTES);
    }
    {
        auto* connectTimeout = conf.MutableConnectTimeout();
        connectTimeout->SetValue(5);
        connectTimeout->SetUnit(TimeUnit::SECONDS);
    }
    {
        auto* readTimeout = conf.MutableReadTimeout();
        readTimeout->SetValue(25);
        readTimeout->SetUnit(TimeUnit::SECONDS);
    }
    return conf;
}

} // namespace

void TStockpileError::Out(IOutputStream& out) const {
    out << "{ rpcCode: " << NGrpc::StatusCodeToString(RpcCode);
    out << ", stockpileCode: " << EStockpileStatusCode_Name(StockpileCode);

    if (Message.empty()) {
        out << " }";
    } else {
        out << ", message: '" << Message << "' }";
    }
}

/**
 * Single node Stockpile RPC.
 */
IStockpileRpcPtr CreateStockpileRpc(const TString& address, NMonitoring::TMetricRegistry& registry, TString clientId) {
    auto conf = CreateConfig(address);
    auto threadPool = CreateGrpcThreadPool(conf);
    auto sc = CreateGrpcServiceConnection<StockpileService>(
        conf,
        false,
        registry,
        std::move(threadPool),
        std::move(clientId));

    return CreateStockpileClient(registry, std::move(sc));
}

/**
 * Cluster aware Stockpile RPC.
 */
IStockpileClusterRpcPtr CreateStockpileRpc(
        const yandex::solomon::config::rpc::TGrpcClientConfig& conf,
        NMonitoring::TMetricRegistry& registry, TString clientId)
{
    return std::make_shared<TGrpcClusterClientBase<StockpileService, IStockpileRpc>>(
        conf,
        false,
        registry,
        CreateStockpileClient,
        std::move(clientId));
}

} // namespace NSolomon::NDataProxy

template <>
void Out<NSolomon::NDataProxy::TStockpileError>(IOutputStream& o, const NSolomon::NDataProxy::TStockpileError& e) {
    e.Out(o);
}
