#pragma once

#include <solomon/libs/cpp/error_or/error_or.h>
#include <solomon/libs/cpp/grpc/client/client.h>

#include <solomon/protos/configs/rpc/rpc_config.pb.h>
#include <solomon/protos/stockpile/status_code.pb.h>
#include <solomon/protos/stockpile/stockpile_read.pb.h>
#include <solomon/protos/stockpile/stockpile_requests.pb.h>

#include <library/cpp/monlib/metrics/fwd.h>
#include <library/cpp/threading/future/future.h>

#include <grpc++/support/status_code_enum.h>

namespace NSolomon::NDataProxy {

/**
 * RPC or Stockpile error.
 */
struct TStockpileError {
    grpc::StatusCode RpcCode{grpc::StatusCode::UNKNOWN};
    yandex::solomon::stockpile::EStockpileStatusCode StockpileCode{yandex::solomon::stockpile::EStockpileStatusCode::UNKNOWN};
    TString Message;

    TStockpileError() = default;

    TStockpileError(
            grpc::StatusCode rpcCode,
            yandex::solomon::stockpile::EStockpileStatusCode stockpileCode,
            TString message)
        : RpcCode(rpcCode)
        , StockpileCode(stockpileCode)
        , Message(std::move(message))
    {
    }

    void Out(IOutputStream& out) const;
};

template <typename TResp>
using TStockpileAsyncResponse = NThreading::TFuture<TErrorOr<TResp, TStockpileError>>;

using TAsyncStockpileStatusResponse = TStockpileAsyncResponse<yandex::solomon::stockpile::TServerStatusResponse>;
using TAsyncReadResponse = TStockpileAsyncResponse<yandex::solomon::stockpile::TReadResponse>;
using TAsyncUncompressedReadManyResponse = TStockpileAsyncResponse<yandex::solomon::stockpile::TUncompressedReadManyResponse>;
using TAsyncCompressedReadResponse = TStockpileAsyncResponse<yandex::solomon::stockpile::TCompressedReadResponse>;
using TAsyncCompressedReadManyResponse = TStockpileAsyncResponse<yandex::solomon::stockpile::TCompressedReadManyResponse>;
using TAsyncReadMetricsMetaResponse = TStockpileAsyncResponse<yandex::solomon::stockpile::ReadMetricsMetaResponse>;

/**
 * Interface of single node Stockpile RPC.
 *
 * TODO: add all methods
 */
class IStockpileRpc: public TThrRefBase {
public:
    virtual TAsyncStockpileStatusResponse ServerStatus(const yandex::solomon::stockpile::TServerStatusRequest& request) = 0;
    virtual TAsyncReadResponse ReadOne(const yandex::solomon::stockpile::TReadRequest& request) = 0;
    virtual TAsyncUncompressedReadManyResponse ReadUncompressedMany(const yandex::solomon::stockpile::TReadManyRequest& request) = 0;
    virtual TAsyncCompressedReadResponse ReadCompressedOne(const yandex::solomon::stockpile::TReadRequest& request) = 0;
    virtual TAsyncCompressedReadManyResponse ReadCompressedMany(const yandex::solomon::stockpile::TReadManyRequest& request) = 0;
    virtual TAsyncReadMetricsMetaResponse ReadMetricsMeta(const yandex::solomon::stockpile::ReadMetricsMetaRequest& request) = 0;
};
using IStockpileRpcPtr = std::unique_ptr<IStockpileRpc>;
using IStockpileClusterRpc = IClusterRpc<IStockpileRpc>;
using IStockpileClusterRpcPtr = std::shared_ptr<IStockpileClusterRpc>;

/**
 * Creates single node Stockpile RPC.
 */
IStockpileRpcPtr CreateStockpileRpc(const TString& address, NMonitoring::TMetricRegistry& registry);

/**
 * Creates Stockpile RPC for each cluster node.
 */
IStockpileClusterRpcPtr CreateStockpileRpc(
        const yandex::solomon::config::rpc::TGrpcClientConfig& conf,
        NMonitoring::TMetricRegistry& registry,
        TString clientId = {});

} // namespace NSolomon::NDataProxy
