#include "rpc_stub.h"

#include <util/generic/hash.h>

namespace NSolomon::NDataProxy {
namespace {

using namespace yandex::solomon::stockpile;

/**
 * Stub single node RPC implementation.
 */
class TStubStockpileRpc: public IStockpileRpc {
public:
    explicit TStubStockpileRpc(TStubStockpileHandlers handlers)
        : Handlers_{std::move(handlers)}
    {
    }

    TAsyncStockpileStatusResponse ServerStatus(const TServerStatusRequest& req) override {
        if (Handlers_.OnServerStatus) {
            return Handlers_.OnServerStatus(req);
        }
        return StockpileFailedResponse<TAsyncStockpileStatusResponse>("not implemented");
    }

    TAsyncReadResponse ReadOne(const yandex::solomon::stockpile::TReadRequest& req) override {
        if (Handlers_.OnReadOne) {
            return Handlers_.OnReadOne(req);
        }
        return StockpileFailedResponse<TAsyncReadResponse>("not implemented");
    }

    TAsyncUncompressedReadManyResponse ReadUncompressedMany(const TReadManyRequest& req) override {
        if (Handlers_.OnReadUncompressedMany) {
            return Handlers_.OnReadUncompressedMany(req);
        }
        return StockpileFailedResponse<TAsyncUncompressedReadManyResponse>("not implemented");
    }

    TAsyncCompressedReadResponse ReadCompressedOne(const TReadRequest& req) override {
        if (Handlers_.OnReadCompressedOne) {
            return Handlers_.OnReadCompressedOne(req);
        }
        return StockpileFailedResponse<TAsyncCompressedReadResponse>("not implemented");
    }

    TAsyncCompressedReadManyResponse ReadCompressedMany(const TReadManyRequest& req) override {
        if (Handlers_.OnReadCompressedMany) {
            return Handlers_.OnReadCompressedMany(req);
        }
        return StockpileFailedResponse<TAsyncCompressedReadManyResponse>("not implemented");
    }

    TAsyncReadMetricsMetaResponse ReadMetricsMeta(const ReadMetricsMetaRequest& req) override {
        if (Handlers_.OnReadMetricsMeta) {
            return Handlers_.OnReadMetricsMeta(req);
        }
        return StockpileFailedResponse<TAsyncReadMetricsMetaResponse>("not implemented");
    }

private:
    TStubStockpileHandlers Handlers_;
};

/**
 * Stub cluster RPC implementation.
 */
class TStubStockpileClusterRpc: public IStockpileClusterRpc {
public:
    explicit TStubStockpileClusterRpc(const THashMap<TString, TStubStockpileHandlers>& nodes) {
        Addresses_.reserve(Nodes_.size());

        for (auto& [address, handlers]: nodes) {
            auto [it, inserted] = Nodes_.emplace(address, std::make_unique<TStubStockpileRpc>(std::move(handlers)));
            Y_ENSURE(inserted, "duplicated address: " << address);
            Addresses_.push_back(it->first);
        }
    }

    IStockpileRpc* Get(TStringBuf address) noexcept override {
        auto it = Nodes_.find(address);
        if (it == Nodes_.end()) {
            return nullptr;
        }
        return it->second.get();
    }

    IStockpileRpc* GetAny() noexcept override {
        if (Nodes_.empty()) {
            return nullptr;
        }
        return Nodes_.begin()->second.get();
    }

    TString GetAnyAddress() const noexcept override {
        if (Nodes_.empty()) {
            return {};
        }

        return Nodes_.begin()->first;
    }

    const std::vector<TString>& Addresses() const noexcept override {
        return Addresses_;
    }

    void Add(TStringBuf) override {
        ythrow yexception() << "not implemented";
    }

    void Stop(bool) override {
    }

private:
    THashMap<TString, IStockpileRpcPtr> Nodes_;
    TVector<TString> Addresses_;
};

} // namespace

// NOLINTNEXTLINE(performance-unnecessary-value-param): false positive
IStockpileClusterRpcPtr StubStockpileRpc(THashMap<TString, TStubStockpileHandlers> nodes) {
    return std::make_shared<TStubStockpileClusterRpc>(std::move(nodes));
}

} // namespace NSolomon::NDataProxy
