#include <solomon/services/dataproxy/lib/stockpile/events.h>
#include <solomon/services/dataproxy/lib/stockpile/shard_actor.h>
#include <solomon/services/dataproxy/lib/stockpile/stub/rpc_stub.h>

#include <solomon/services/dataproxy/config/datasource_config.pb.h>

#include <solomon/libs/cpp/actors/test_runtime/actor_runtime.h>

#include <library/cpp/actors/core/event.h>
#include <library/cpp/testing/gtest/gtest.h>

using namespace NSolomon;
using namespace NDataProxy;
using namespace yandex::solomon::stockpile;

class TStockpileShardActorTest: public ::testing::Test {
protected:
    void SetUp() override {
        ActorRuntime_ = TTestActorRuntime::CreateInited(1, false, true);
        ActorRuntime_->WaitForBootstrap();
        EdgeId_ = ActorRuntime_->AllocateEdgeActor();
    }

    void TearDown() override {
        ActorRuntime_.Reset();
    }

    template <typename TEvent>
    typename TEvent::TPtr ReceiveResponse() {
        return ActorRuntime_->GrabEdgeEvent<TEvent>(EdgeId_, TDuration::Seconds(10));
    }

protected:
    THolder<TTestActorRuntime> ActorRuntime_;
    NActors::TActorId EdgeId_;
};

TEST_F(TStockpileShardActorTest, ReadOne_Success) {
    THashMap<TString, TStubStockpileHandlers> handlers;
    handlers["node1"] = TStubStockpileHandlers{
            .OnReadCompressedOne = [count = 0u](const TReadRequest& req) mutable {
                Y_ENSURE(++count == 1); // no retries

                Y_ENSURE(req.metric_id().shard_id() == 42);
                Y_ENSURE(req.metric_id().local_id() == 157);

                TCompressedReadResponse resp;
                resp.set_status(EStockpileStatusCode::OK);
                resp.mutable_metric_id()->set_shard_id(42);
                resp.mutable_metric_id()->set_local_id(157);
                return StockpileResponse(resp);
            }
    };

    ui64 cookie = 1337;

    auto rpc = StubStockpileRpc(std::move(handlers));
    auto requesterId = ActorRuntime_->Register(StockpileShardActor(rpc, TStockpileShardInfo{42, true, "node1"}, 2000).release());

    auto req = std::make_unique<TStockpileEvents::TReadOneReq>();
    req->Deadline = TDuration::Seconds(15).ToDeadLine();
    req->Message.set_deadline(req->Deadline.MilliSeconds());
    req->Message.mutable_metric_id()->set_shard_id(42);
    req->Message.mutable_metric_id()->set_local_id(157);
    ActorRuntime_->Send(new NActors::IEventHandle(requesterId, EdgeId_, req.release(), 0, cookie));

    auto resp = ReceiveResponse<TStockpileEvents::TReadOneResp>();
    ASSERT_TRUE(resp);
    EXPECT_EQ(resp->Cookie, cookie);
    EXPECT_EQ(resp->Get()->Message.status(), EStockpileStatusCode::OK);
    EXPECT_EQ(resp->Get()->Message.metric_id().shard_id(), 42u);
    EXPECT_EQ(resp->Get()->Message.metric_id().local_id(), 157u);

    ASSERT_FALSE(!ActorRuntime_->FindActor(requesterId));
}

TEST_F(TStockpileShardActorTest, ReadOne_RetryableError) {
    THashMap<TString, TStubStockpileHandlers> handlers;
    handlers["node1"] = TStubStockpileHandlers{
            .OnReadCompressedOne = [count = 0u](const TReadRequest&) mutable {
                TCompressedReadResponse resp;
                ++count;
                if (count < 4) {
                    resp.set_status(EStockpileStatusCode::NODE_UNAVAILABLE);
                } else if (count == 4) {
                    resp.set_status(EStockpileStatusCode::OK);
                } else {
                    Y_FAIL("unexpected number of retries");
                }
                return StockpileResponse(resp);
            }
    };

    ui64 cookie = 1337;

    auto rpc = StubStockpileRpc(std::move(handlers));
    auto requesterId = ActorRuntime_->Register(StockpileShardActor(rpc, TStockpileShardInfo{42, true, "node1"}, 2000).release());

    auto req = std::make_unique<TStockpileEvents::TReadOneReq>();
    req->Message.set_deadline(TDuration::Seconds(15).ToDeadLine().MilliSeconds());
    ActorRuntime_->Send(new NActors::IEventHandle(requesterId, EdgeId_, req.release(), 0, cookie));

    auto resp = ReceiveResponse<TStockpileEvents::TReadOneResp>();
    ASSERT_TRUE(resp);
    EXPECT_EQ(resp->Cookie, cookie);
    EXPECT_EQ(resp->Get()->Message.status(), EStockpileStatusCode::OK);

    ASSERT_FALSE(!ActorRuntime_->FindActor(requesterId));
}

TEST_F(TStockpileShardActorTest, ReadOne_NonRetryableError) {
    THashMap<TString, TStubStockpileHandlers> handlers;
    handlers["node1"] = TStubStockpileHandlers{
            .OnReadCompressedOne = [count = 0u](const TReadRequest&) mutable {
                Y_ENSURE(++count == 1); // no retries

                TCompressedReadResponse resp;
                resp.set_status(EStockpileStatusCode::INVALID_REQUEST);
                resp.set_statusmessage("some message");
                return StockpileResponse(resp);
            }
    };

    ui64 cookie = 1337;

    auto rpc = StubStockpileRpc(std::move(handlers));
    auto requesterId = ActorRuntime_->Register(StockpileShardActor(rpc, TStockpileShardInfo{42, true, "node1"}, 2000).release());

    auto req = std::make_unique<TStockpileEvents::TReadOneReq>();
    req->Message.set_deadline(TDuration::Seconds(15).ToDeadLine().MilliSeconds());
    ActorRuntime_->Send(new NActors::IEventHandle(requesterId, EdgeId_, req.release(), 0, cookie));

    auto resp = ReceiveResponse<TStockpileEvents::TError>();
    ASSERT_TRUE(resp);
    EXPECT_EQ(resp->Cookie, cookie);
    EXPECT_EQ(resp->Get()->RpcCode, grpc::StatusCode::OK);
    EXPECT_EQ(resp->Get()->StockpileCode, EStockpileStatusCode::INVALID_REQUEST);
    EXPECT_EQ(resp->Get()->Message, "some message [node1]");

    ASSERT_FALSE(!ActorRuntime_->FindActor(requesterId));
}

TEST_F(TStockpileShardActorTest, ReadOne_ErrorResponse) {
    THashMap<TString, TStubStockpileHandlers> handlers;
    handlers["node1"] = TStubStockpileHandlers{
            .OnReadCompressedOne = [count = 0u](const TReadRequest&) mutable {
                Y_ENSURE(++count == 1); // no retries
                return StockpileErrorResponse<TCompressedReadResponse>(
                        grpc::StatusCode::INVALID_ARGUMENT,
                        EStockpileStatusCode::UNKNOWN,
                        TString{"some message"});
            }
    };

    ui64 cookie = 1337;

    auto rpc = StubStockpileRpc(std::move(handlers));
    auto requesterId = ActorRuntime_->Register(StockpileShardActor(rpc, TStockpileShardInfo{42, true, "node1"}, 2000).release());

    auto req = std::make_unique<TStockpileEvents::TReadOneReq>();
    req->Message.set_deadline(TDuration::Seconds(15).ToDeadLine().MilliSeconds());
    ActorRuntime_->Send(new NActors::IEventHandle(requesterId, EdgeId_, req.release(), 0, cookie));

    auto resp = ReceiveResponse<TStockpileEvents::TError>();
    ASSERT_TRUE(resp);
    EXPECT_EQ(resp->Cookie, cookie);
    EXPECT_EQ(resp->Get()->RpcCode, grpc::StatusCode::INVALID_ARGUMENT);
    EXPECT_EQ(resp->Get()->StockpileCode, EStockpileStatusCode::UNKNOWN);
    EXPECT_EQ(resp->Get()->Message, "some message [node1]");

    ASSERT_FALSE(!ActorRuntime_->FindActor(requesterId));
}
