#include <solomon/services/dataproxy/lib/stockpile/stub/rpc_stub.h>
#include <solomon/services/dataproxy/lib/stockpile/watcher.h>

#include <solomon/libs/cpp/actors/test_runtime/actor_runtime.h>

#include <library/cpp/actors/core/event.h>
#include <library/cpp/testing/gtest/gtest.h>

using namespace NSolomon;
using namespace NDataProxy;

using namespace yandex::solomon::stockpile;
using namespace yandex::solomon::model;

namespace {

class TFakeStockpileNode {
public:
    TFakeStockpileNode(TString address, std::initializer_list<TStockpileShardId> shardIds)
        : Address_{std::move(address)}
    {
        for (TStockpileShardId id: shardIds) {
            Shards_[id] = true;
        }
    }

    void AddShard(ui16 shardId) {
        Shards_[shardId] = true;
    }

    void ChangeShardState(TStockpileShardId shardId, bool ready) {
        if (auto it = Shards_.find(shardId); it != Shards_.end()) {
            it->second = ready;
        }
    }

    void RemoveShard(TStockpileShardId shardId) {
        Shards_.erase(shardId);
    }

    TServerStatusResponse Status() const {
        TServerStatusResponse resp;
        resp.SetStatus(EStockpileStatusCode::OK);

        for (auto& [shardId, ready]: Shards_) {
            TShardStatus* shard = resp.AddShardStatus();
            shard->SetShardId(shardId);
            shard->SetReady(ready);
        }

        resp.SetTotalShardCount(Shards_.size());

        resp.SetOlderSupportBinaryVersion(10);
        resp.SetLatestSupportBinaryVersion(10);
        return resp;
    }

    const TString& Address() const noexcept {
        return Address_;
    }

private:
    TString Address_;
    std::map<ui16, bool> Shards_; // id -> ready
};

IStockpileClusterRpcPtr MakeFakeRpc(const std::vector<TFakeStockpileNode*>& nodes) {
    THashMap<TString, TStubStockpileHandlers> handlers;
    for (TFakeStockpileNode* node: nodes) {
        handlers[node->Address()] = TStubStockpileHandlers{
            .OnServerStatus = [node](const TServerStatusRequest& req) {
                Y_ENSURE(req.deadline() != 0, "deadline is not set");
                return StockpileResponse(node->Status());
            }
        };
    }
    return StubStockpileRpc(std::move(handlers));
}

TStockpileShardsMap ToShardsMap(const std::vector<TStockpileShardInfo>& shards) {
    TStockpileShardsMap shardsMap;
    for (const auto& s: shards) {
        shardsMap[s.Id] = s;
    }
    return shardsMap;
}

} // namespace

class TWatcherTest: public ::testing::Test {
protected:
    void SetUp() override {
        ActorRuntime_ = TTestActorRuntime::CreateInited(1, false, true);
        ActorRuntime_->WaitForBootstrap();
        EdgeId_ = ActorRuntime_->AllocateEdgeActor();
    }

    void TearDown() override {
        ActorRuntime_.Reset();
    }

    template <typename TEvent>
    typename TEvent::TPtr ReceiveResponse() {
        return ActorRuntime_->GrabEdgeEvent<TEvent>(EdgeId_, TDuration::Seconds(10));
    }

protected:
    THolder<TTestActorRuntime> ActorRuntime_;
    NActors::TActorId EdgeId_;
};

TEST_F(TWatcherTest, Subscribe) {
    TFakeStockpileNode nodeA{"node-a", {1}};
    TFakeStockpileNode nodeB{"node-b", {100, 200}};
    auto rpc = MakeFakeRpc({&nodeA, &nodeB});

    auto watcher = StockpileClusterWatcher(rpc, rpc->Addresses(), TDuration::Seconds(1));
    auto watcherId = ActorRuntime_->Register(watcher.release());
    ActorRuntime_->WaitForBootstrap();

    ActorRuntime_->Send(watcherId, EdgeId_, MakeHolder<TStockpileWatcherEvents::TSubscribe>());

    {
        auto update = ReceiveResponse<TStockpileWatcherEvents::TUpdate>();
        ASSERT_TRUE(update);

        auto shards = ToShardsMap(update->Get()->Shards);
        EXPECT_EQ(shards.size(), 3u);

        EXPECT_TRUE(shards[1].Ready);
        EXPECT_EQ(shards[1].Location, "node-a");

        EXPECT_TRUE(shards[100].Ready);
        EXPECT_EQ(shards[100].Location, "node-b");

        EXPECT_TRUE(shards[200].Ready);
        EXPECT_EQ(shards[200].Location, "node-b");
    }

    // disable shard 1 and add new one
    nodeA.ChangeShardState(1, false);
    nodeA.AddShard(2);
    ActorRuntime_->AdvanceCurrentTime(TDuration::Seconds(10));

    {
        auto update = ReceiveResponse<TStockpileWatcherEvents::TUpdate>();
        ASSERT_TRUE(update);

        auto shards = ToShardsMap(update->Get()->Shards);
        EXPECT_EQ(shards.size(), 2u);

        EXPECT_FALSE(shards[1].Ready);
        EXPECT_EQ(shards[1].Location, "node-a");

        EXPECT_TRUE(shards[2].Ready);
        EXPECT_EQ(shards[2].Location, "node-a");
    }

    // move shard 2 from node a to node b
    nodeA.RemoveShard(2);
    nodeB.AddShard(2);
    ActorRuntime_->AdvanceCurrentTime(TDuration::Seconds(10));

    {
        auto update = ReceiveResponse<TStockpileWatcherEvents::TUpdate>();
        ASSERT_TRUE(update);

        auto shards = ToShardsMap(update->Get()->Shards);
        EXPECT_EQ(shards.size(), 1u);

        EXPECT_TRUE(shards[2].Ready);
        EXPECT_EQ(shards[2].Location, "node-b");
    }

    // enable shard 1
    nodeA.ChangeShardState(1, true);
    ActorRuntime_->AdvanceCurrentTime(TDuration::Seconds(10));

    {
        auto update = ReceiveResponse<TStockpileWatcherEvents::TUpdate>();
        ASSERT_TRUE(update);

        auto shards = ToShardsMap(update->Get()->Shards);
        EXPECT_EQ(shards.size(), 1u);

        EXPECT_TRUE(shards[1].Ready);
        EXPECT_EQ(shards[1].Location, "node-a");
    }
}

TEST_F(TWatcherTest, Resolve) {
    TFakeStockpileNode nodeA{"node-a", {1}};
    TFakeStockpileNode nodeB{"node-b", {100, 200}};
    auto rpc = MakeFakeRpc({&nodeA, &nodeB});

    auto watcher = StockpileClusterWatcher(rpc, rpc->Addresses(), TDuration::Seconds(1));
    auto watcherId = ActorRuntime_->Register(watcher.release());
    ActorRuntime_->WaitForBootstrap();

    {
        auto event = std::make_unique<TStockpileWatcherEvents::TResolve>();
        event->Ids.push_back(100);
        event->Ids.push_back(1);
        ActorRuntime_->Send(watcherId, EdgeId_, THolder(event.release()));

        auto result = ReceiveResponse<TStockpileWatcherEvents::TResolveResult>();
        ASSERT_TRUE(result);
        EXPECT_EQ(result->Get()->Shards.size(), 2u);
        EXPECT_EQ(result->Get()->NotFound.size(), 0u);

        auto shards = ToShardsMap(result->Get()->Shards);
        EXPECT_TRUE(shards[1].Ready);
        EXPECT_EQ(shards[1].Location, "node-a");

        EXPECT_TRUE(shards[100].Ready);
        EXPECT_EQ(shards[100].Location, "node-b");
    }

    {
        auto event = std::make_unique<TStockpileWatcherEvents::TResolve>();
        event->Ids.push_back(200);
        event->Ids.push_back(300);
        ActorRuntime_->Send(watcherId, EdgeId_, THolder(event.release()));

        auto result = ReceiveResponse<TStockpileWatcherEvents::TResolveResult>();
        ASSERT_TRUE(result);
        EXPECT_EQ(result->Get()->Shards.size(), 1u);
        EXPECT_EQ(result->Get()->NotFound.size(), 1u);

        auto shards = ToShardsMap(result->Get()->Shards);
        EXPECT_TRUE(shards[200].Ready);
        EXPECT_EQ(shards[200].Location, "node-b");

        EXPECT_EQ(result->Get()->NotFound[0], 300u);
    }
}
