#pragma once

#include "timeseries.h"
#include "type_to_columns.h"
#include "compression.h"

#include <solomon/libs/cpp/ts_codec/bit_buffer.h>

#include <util/generic/buffer.h>

namespace NSolomon::NDataProxy {

/**
 * Compressed time series.
 */
class TCompressedTimeSeries: public ITimeSeries {
public:
    TCompressedTimeSeries(
            NMonitoring::EMetricType type,
            NTs::TColumnSet columns,
            ui32 pointCount,
            NTs::TBitBuffer buffer)
        : Type_{type}
        , Columns_{columns}
        , PointCount_{pointCount}
        , Buffer_{std::move(buffer)}
    {
        EnsureValidTypeForColumns(Type_, Columns_);
    }

    TCompressedTimeSeries(NMonitoring::EMetricType type, NTs::TColumnSet columns, ITimeSeriesIter& it)
        : Type_{type}
        , Columns_{columns}
    {
        EnsureValidTypeForColumns(Type_, Columns_);

        auto [buf, pointCount] = Compress(type, columns, it);
        Buffer_ = std::move(buf);
        PointCount_ = pointCount;
    }

    explicit TCompressedTimeSeries(const ITimeSeries& rhs)
        : Type_{rhs.Type()}
        , Columns_{rhs.Columns()}
    {
        EnsureValidTypeForColumns(Type_, Columns_);

        auto [buf, pointCount] = Compress(rhs);
        Buffer_ = std::move(buf);
        PointCount_ = pointCount;
    }

    NMonitoring::EMetricType Type() const override {
        return Type_;
    }

    NTs::TColumnSet Columns() const override {
        return Columns_;
    }

    ui32 PointCount() const override {
        return PointCount_;
    }

    std::unique_ptr<ITimeSeriesIter> Iterator() const override {
        return Decompress(Type_, Columns_, Buffer_);
    }

    NTs::TBitSpan Buffer() const & {
        return Buffer_;
    }

    NTs::TBitBuffer Buffer() && {
        return std::move(Buffer_);
    }

private:
    NMonitoring::EMetricType Type_;
    NTs::TColumnSet Columns_;
    ui32 PointCount_;
    NTs::TBitBuffer Buffer_;
};

// TODO: implement mutable compressed time series

} // namespace NSolomon::NDataProxy
