#include "compression.h"
#include "empty.h"

#include <solomon/libs/cpp/ts_codec/double_ts_codec.h>
#include <solomon/libs/cpp/ts_codec/counter_ts_codec.h>
#include <solomon/libs/cpp/ts_codec/gauge_int_ts_codec.h>
#include <solomon/libs/cpp/ts_codec/hist_ts_codec.h>
#include <solomon/libs/cpp/ts_codec/hist_log_ts_codec.h>
#include <solomon/libs/cpp/ts_codec/summary_double_ts_codec.h>

namespace NSolomon::NDataProxy {
namespace {

template <typename TDecoder>
class TDecodeIter final: public ITimeSeriesIter {
public:
    TDecodeIter(NTs::TColumnSet columns, NTs::TBitSpan data)
        : Decoder_{columns, data}
    {
    }

    bool Next(NTs::TVariantPoint* point) override {
        return Decoder_.NextPoint(point);
    }

private:
    TDecoder Decoder_;
};

template <typename TEncoder>
std::pair<NTs::TBitBuffer, ui32> Encode(NTs::TColumnSet columns, ITimeSeriesIter& it) {
    NTs::TBitBuffer buf;
    NTs::TBitWriter w{&buf};
    TEncoder encoder{columns, &w};

    NTs::TVariantPoint point;
    while (it.Next(&point)) {
        encoder.EncodePoint(point);
    }

    encoder.Flush();
    ui32 pointCount = encoder.FrameSize();
    return {std::move(buf), pointCount};
}

} // namespace

std::pair<NTs::TBitBuffer, ui32> Compress(const ITimeSeries& timeSeries) {
    NMonitoring::EMetricType type = timeSeries.Type();
    NTs::TColumnSet columns = timeSeries.Columns();
    auto it = timeSeries.Iterator();
    return Compress(type, columns, *it);
}

std::pair<NTs::TBitBuffer, ui32> Compress(NMonitoring::EMetricType type, NTs::TColumnSet columns, ITimeSeriesIter& it) {
    switch (type) {
        case NMonitoring::EMetricType::GAUGE:
            return Encode<NTs::TDoubleTsEncoder>(columns, it);

        case NMonitoring::EMetricType::COUNTER:
        case NMonitoring::EMetricType::RATE:
            return Encode<NTs::TCounterTsEncoder>(columns, it);

        case NMonitoring::EMetricType::IGAUGE:
            return Encode<NTs::TGaugeIntTsEncoder>(columns, it);

        case NMonitoring::EMetricType::HIST:
        case NMonitoring::EMetricType::HIST_RATE:
            return Encode<NTs::THistogramTsEncoder>(columns, it);

        case NMonitoring::EMetricType::DSUMMARY:
            return Encode<NTs::TSummaryDoubleTsEncoder>(columns, it);

        case NMonitoring::EMetricType::LOGHIST:
            return Encode<NTs::TLogHistogramTsEncoder>(columns, it);

        case NMonitoring::EMetricType::UNKNOWN: {
            if (NTs::TVariantPoint point; it.Next(&point)) {
                ythrow yexception() << "cannot compress non empty time series with unknown type";
            }
            return {NTs::TBitBuffer{}, 0};
        }
    }
}

std::unique_ptr<ITimeSeriesIter> Decompress(NMonitoring::EMetricType type, NTs::TColumnSet columns, NTs::TBitSpan data) {
    switch (type) {
        case NMonitoring::EMetricType::GAUGE:
            return std::make_unique<TDecodeIter<NTs::TDoubleTsDecoder>>(columns, data);

        case NMonitoring::EMetricType::COUNTER:
        case NMonitoring::EMetricType::RATE:
            return std::make_unique<TDecodeIter<NTs::TCounterTsDecoder>>(columns, data);

        case NMonitoring::EMetricType::IGAUGE:
            return std::make_unique<TDecodeIter<NTs::TGaugeIntTsDecoder>>(columns, data);

        case NMonitoring::EMetricType::HIST:
        case NMonitoring::EMetricType::HIST_RATE:
            return std::make_unique<TDecodeIter<NTs::THistogramTsDecoder>>(columns, data);

        case NMonitoring::EMetricType::DSUMMARY:
            return std::make_unique<TDecodeIter<NTs::TSummaryDoubleTsDecoder>>(columns, data);

        case NMonitoring::EMetricType::LOGHIST:
            return std::make_unique<TDecodeIter<NTs::TLogHistogramTsDecoder>>(columns, data);

        case NMonitoring::EMetricType::UNKNOWN: {
            Y_ENSURE(data.Size() == 0, "cannot decompress nonempty metric with unknown type");
            return std::make_unique<TEmptyTimeSeriesIter>();
        }
    }
}

} // namespace NSolomon::NDataProxy
