#pragma once

#include "timeseries.h"

#include <solomon/libs/cpp/ts_model/merge.h>

namespace NSolomon::NDataProxy {

/**
 * Merges two points together by using only one value with the maximum count.
 */
struct TMaxMerger {
    static void Merge(NTs::TVariantPoint* dst, NTs::TVariantPoint* src) {
        if (src->Count > dst->Count) {
            *dst = std::move(*src);
        }
    }
};

/**
 * Merges two points together by combining their values field by field.
 * See MergeRaw() implementation for more information.
 */
struct TCombineMerger {
    static void Merge(NTs::TVariantPoint* dst, NTs::TVariantPoint* src) {
        if (dst->Merge) {
            std::visit([src](auto&& arg) {
                NTsModel::MergeRaw(src->Get<std::decay_t<decltype(arg)>>(), &arg);
            }, dst->Value);
            dst->Count += src->Count;
        }
    }
};

template <typename TMerger>
class TMergingIterator final: public ITimeSeriesIter {
    struct TIterState {
        std::unique_ptr<ITimeSeriesIter> Iter;
        NTs::TVariantPoint Point;

        explicit TIterState(std::unique_ptr<ITimeSeriesIter> iter)
            : Iter{std::move(iter)}
        {
        }

        bool HasNext(NTs::TVariantPoint* target, TInstant latestTime) {
            if (Point.Time <= latestTime) {
                if (!Iter->Next(&Point)) {
                    Point.Time = TInstant::Zero();
                    return false;
                }
            }

            if (target->Time == TInstant::Zero() || target->Time > Point.Time) {
                *target = std::move(Point);
                return true;
            }

            if (target->Time == Point.Time) {
                TMerger::Merge(target, &Point);
                return true;
            }

            return true;
        }
    };

public:
    void AddTimeSeries(const ITimeSeries& timeSeries) {
        Columns_ = Columns_ | timeSeries.Columns();
        Iterators_.emplace_back(timeSeries.Iterator());
    }

    bool Next(NTs::TVariantPoint* target) override {
        bool hasNext = false;
        target->Time = TInstant::Zero();

        for (TIterState& state: Iterators_) {
            hasNext |= state.HasNext(target, LatestTime_);
        }

        if (hasNext) {
            LatestTime_ = target->Time;
        }

        return hasNext;
    }

    NTs::TColumnSet Columns() const {
        return Columns_;
    }

    bool Empty() const noexcept {
        return Iterators_.empty();
    }

private:
    TInstant LatestTime_;
    NTs::TColumnSet Columns_;
    std::vector<TIterState> Iterators_;
};

std::unique_ptr<ITimeSeries> MergeTimeSeries(
    NMonitoring::EMetricType resultType,
    std::vector<std::unique_ptr<ITimeSeries>> timeSeries);

} // namespace NSolomon::NDataProxy
