#include "protobuf.h"
#include "empty.h"

#include <solomon/libs/cpp/proto_convert/metric_type.h>
#include <solomon/libs/cpp/stockpile_codec/format.h>
#include <solomon/libs/cpp/stockpile_codec/metric_archive.h>
#include <solomon/libs/cpp/ts_codec/counter_ts_codec.h>
#include <solomon/libs/cpp/ts_codec/double_ts_codec.h>
#include <solomon/libs/cpp/ts_codec/gauge_int_ts_codec.h>
#include <solomon/libs/cpp/ts_codec/hist_log_ts_codec.h>
#include <solomon/libs/cpp/ts_codec/hist_ts_codec.h>
#include <solomon/libs/cpp/ts_codec/summary_double_ts_codec.h>

namespace NSolomon::NDataProxy {
namespace {

struct TChunk: public TIntrusiveSListItem<TChunk> {
    TInstant From;       // inclusive
    TInstant To;         // exclusive
    NStockpile::TMetricArchive Archive;

    explicit TChunk(NStockpile::EFormat format)
        : Archive{format}
    {
    }
};

template <typename TDecoder>
class TChunkedIter: public ITimeSeriesIter {
    using TChunkIt = std::vector<TChunk>::const_iterator;
public:
    TChunkedIter(TChunkIt it, TChunkIt end)
        : It_{it}
        , End_{end}
    {
        CreateDecoder();
    }

    void CreateDecoder() {
        Y_VERIFY(It_ != End_);

        // TODO: allow to reset decoder state, to avoid entire decoder allocation on a heap
        Decoder_ = std::make_unique<TDecoder>(It_->Archive.Columns(), It_->Archive.Data());

        ++It_;
    }

    bool Next(NTs::TVariantPoint* point) override {
        if (Decoder_->NextPoint(point)) {
            return true;
        }

        if (It_ == End_) {
            return false;
        }

        CreateDecoder();
        return Decoder_->NextPoint(point);
    }

private:
    TChunkIt It_;
    TChunkIt End_;
    std::unique_ptr<TDecoder> Decoder_;
};

class TChunkedTimeSeries: public ITimeSeries {
public:
    TChunkedTimeSeries(NMonitoring::EMetricType type, std::vector<TChunk> chunks)
        : Type_{type}
        , Chunks_{std::move(chunks)}
    {
    }

    NMonitoring::EMetricType Type() const override {
        return Type_;
    }

    NTs::TColumnSet Columns() const override {
        return Chunks_[0].Archive.Columns();
    }

    ui32 PointCount() const override {
        ui32 count = 0;
        for (const auto& chunk: Chunks_) {
            count += chunk.Archive.PointCount();
        }
        return count;
    }

    std::unique_ptr<ITimeSeriesIter> Iterator() const override {
        switch (Type_) {
            case NMonitoring::EMetricType::GAUGE:
                return std::make_unique<TChunkedIter<NTs::TDoubleTsDecoder>>(Chunks_.begin(), Chunks_.end());

            case NMonitoring::EMetricType::COUNTER:
            case NMonitoring::EMetricType::RATE:
                return std::make_unique<TChunkedIter<NTs::TCounterTsDecoder>>(Chunks_.begin(), Chunks_.end());

            case NMonitoring::EMetricType::IGAUGE:
                return std::make_unique<TChunkedIter<NTs::TGaugeIntTsDecoder>>(Chunks_.begin(), Chunks_.end());

            case NMonitoring::EMetricType::HIST:
            case NMonitoring::EMetricType::HIST_RATE:
                return std::make_unique<TChunkedIter<NTs::THistogramTsDecoder>>(Chunks_.begin(), Chunks_.end());

            case NMonitoring::EMetricType::DSUMMARY:
                return std::make_unique<TChunkedIter<NTs::TSummaryDoubleTsDecoder>>(Chunks_.begin(), Chunks_.end());

            case NMonitoring::EMetricType::LOGHIST:
                return std::make_unique<TChunkedIter<NTs::TLogHistogramTsDecoder>>(Chunks_.begin(), Chunks_.end());

            case NMonitoring::EMetricType::UNKNOWN:
                ythrow yexception() << "cannot parse metric with unknown type";
        }
    }

private:
    NMonitoring::EMetricType Type_;
    std::vector<TChunk> Chunks_;
};

void ParseChunk(NStockpile::EFormat format, TChunk* dstChunk, const yandex::solomon::model::TimeSeries_Chunk& srcChunk) {
    dstChunk->From = TInstant::MilliSeconds(srcChunk.from_millis());
    dstChunk->To = TInstant::MilliSeconds(srcChunk.to_millis());

    NStockpile::TCodecInput input{srcChunk.content()};
    NStockpile::TMetricArchiveCodec codec{format};
    dstChunk->Archive = codec.Decode(&input); // TODO: avoid TBuffer allocation
}

template <typename TProtoMessage>
std::unique_ptr<ITimeSeries> Convert(
    NMonitoring::EMetricType type,
    NStockpile::EFormat format,
    const TProtoMessage& message)
{
    if (message.chunks_size() == 0) {
        return std::make_unique<TEmptyTimeSeries>(type);
    }

    if (message.chunks_size() == 1) {
        TChunk chunk{format};
        ParseChunk(format, &chunk, message.chunks(0));

        if (chunk.Archive.PointCount() == 0) {
            return std::make_unique<TEmptyTimeSeries>(type);
        }

        return std::make_unique<TCompressedTimeSeries>(
            type,
            chunk.Archive.Columns(),
            chunk.Archive.PointCount(),
            std::move(chunk.Archive).Data());
    }

    std::vector<TChunk> chunks(message.chunks_size(), TChunk{format});
    for (int i = 0; i < message.chunks_size(); ++i) {
        ParseChunk(format, &chunks[i], message.chunks(i));
    }

    return std::make_unique<TChunkedTimeSeries>(type, std::move(chunks));
}

} // namespace

void ToProto(const TCompressedTimeSeries& timeSeries, yandex::solomon::model::TimeSeries* proto, NStockpile::EFormat format) {
    NStockpile::TMetricHeader header;
    header.Type = NSolomon::ToProto(timeSeries.Type());

    NStockpile::TMetricArchive archive{
        header,
        format,
        timeSeries.Columns(),
        timeSeries.PointCount(),
        NTs::TBitBuffer(timeSeries.Buffer())};   // TODO: remove redundant copy

    NStockpile::TCodecOutput out{timeSeries.Buffer().SizeBytes()};
    NStockpile::TMetricArchiveCodec codec{format};
    codec.Encode(archive, &out);
    TBuffer archiveBuf = out.TakeBuffer();

    proto->set_format_version(ToUnderlying(format));
    auto* protoChunk = proto->add_chunks();
    protoChunk->set_content(archiveBuf.data(), archiveBuf.size()); // TODO: one more redundant copy
    protoChunk->set_point_count(archive.PointCount());
}

void ToProto(const ITimeSeries& timeSeries, yandex::solomon::model::TimeSeries* proto, NStockpile::EFormat format) {
    if (auto* compressed = dynamic_cast<const TCompressedTimeSeries*>(&timeSeries)) {
        ToProto(*compressed, proto, format);
    } else {
        TCompressedTimeSeries compressedTs{timeSeries};
        ToProto(compressedTs, proto, format);
    }
}

std::unique_ptr<ITimeSeries> FromProto(
        yandex::solomon::model::MetricType type,
        const yandex::solomon::model::TimeSeries& proto)
{
    auto format = NStockpile::FormatFromInt(proto.format_version());
    return Convert(NSolomon::FromProto(type), format, proto);
}

std::unique_ptr<ITimeSeries> FromProto(
        NMonitoring::EMetricType typeFallback,
        const yandex::solomon::stockpile::TCompressedReadResponse& response)
{
    auto type = (response.type() != yandex::solomon::model::MetricType::METRIC_TYPE_UNSPECIFIED)
            ? NSolomon::FromProto(response.type())
            : typeFallback;

    auto format = NStockpile::FormatFromInt(response.GetBinaryVersion());
    return Convert(type, format, response);
}

std::unique_ptr<ITimeSeries> FromProto(
        NMonitoring::EMetricType typeFallback,
        const yandex::solomon::stockpile::MetricData& metricData)
{
    Y_ENSURE(!metricData.HasUncompressed(), "uncompressed data is not supported");

    auto type = (metricData.type() != yandex::solomon::model::MetricType::METRIC_TYPE_UNSPECIFIED)
            ? NSolomon::FromProto(metricData.type())
            : typeFallback;

    if (!metricData.HasCompressed()) {
        return std::make_unique<TEmptyTimeSeries>(type);
    }

    const auto& compressed = metricData.compressed();
    auto format = NStockpile::FormatFromInt(compressed.format_version());
    return Convert(type, format, compressed);
}

} // namespace NSolomon::NDataProxy
