#include <solomon/services/dataproxy/lib/timeseries/aggregates.h>

#include <library/cpp/testing/gtest/gtest.h>

using namespace NSolomon;
using namespace NDataProxy;

using yandex::solomon::math::Aggregate;

TEST(TAggregatesTest, DoubleToProto) {
    TAggregateVariant aggr;
    auto& value = aggr.Value.emplace<TAggregateDouble>();
    value.Max = 1.0;
    value.Min = 2.0;
    value.Sum = 3.0;
    value.Avg = 4.0;
    value.Last = 5.0;
    aggr.Count = 6;
    aggr.Mask = 0b01111110;

    Aggregate proto;
    ToProto(aggr, &proto);

    ASSERT_EQ(proto.type_case(), Aggregate::kDouble);

    auto& aggrProto = proto.double_();
    EXPECT_EQ(aggrProto.max(), value.Max);
    EXPECT_EQ(aggrProto.min(), value.Min);
    EXPECT_EQ(aggrProto.sum(), value.Sum);
    EXPECT_EQ(aggrProto.avg(), value.Avg);
    EXPECT_EQ(aggrProto.last(), value.Last);
    EXPECT_EQ(aggrProto.count(), aggr.Count);
    EXPECT_EQ(aggrProto.mask(), aggr.Mask);
}

TEST(TAggregatesTest, DoubleFromProto) {
    Aggregate proto;
    auto* protoAggr = proto.mutable_double_();
    protoAggr->set_max(1.1);
    protoAggr->set_min(1.2);
    protoAggr->set_sum(1.3);
    protoAggr->set_avg(1.4);
    protoAggr->set_last(1.5);
    protoAggr->set_count(6);
    protoAggr->set_mask(0x00111111);

    TAggregateVariant aggr;
    FromProto(proto, &aggr);

    ASSERT_TRUE(std::holds_alternative<TAggregateDouble>(aggr.Value));
    auto& value = std::get<TAggregateDouble>(aggr.Value);

    EXPECT_EQ(protoAggr->max(), value.Max);
    EXPECT_EQ(protoAggr->min(), value.Min);
    EXPECT_EQ(protoAggr->sum(), value.Sum);
    EXPECT_EQ(protoAggr->avg(), value.Avg);
    EXPECT_EQ(protoAggr->last(), value.Last);
    EXPECT_EQ(protoAggr->count(), aggr.Count);
    EXPECT_EQ(protoAggr->mask(), aggr.Mask);
}

TEST(TAggregatesTest, DoubleConvert) {
    TAggregateVariant aggr1{
        .Value = TAggregateDouble{
            .Max = 1.0,
            .Min = 2.0,
            .Sum = 3.0,
            .Avg = 4.0,
            .Last = 5.0,
        },
        .Count = 6,
        .Mask = 0b01111110,
    };

    Aggregate proto;
    ToProto(aggr1, &proto);

    TAggregateVariant aggr2;
    FromProto(proto, &aggr2);

    EXPECT_EQ(aggr1, aggr2);
}

TEST(TAggregatesTest, Int64ToProto) {
    TAggregateVariant aggr;
    auto& value = aggr.Value.emplace<TAggregateInt64>();
    value.Max = 1;
    value.Min = 2;
    value.Sum = 3;
    value.Avg = 4;
    value.Last = 5;
    aggr.Count = 6;
    aggr.Mask = 0b01111110;

    Aggregate proto;
    ToProto(aggr, &proto);

    ASSERT_EQ(proto.type_case(), Aggregate::kInt64);

    auto& aggrProto = proto.int64();
    EXPECT_EQ(aggrProto.max(), ui64(value.Max));
    EXPECT_EQ(aggrProto.min(), ui64(value.Min));
    EXPECT_EQ(aggrProto.sum(), ui64(value.Sum));
    EXPECT_EQ(aggrProto.avg(), ui64(value.Avg));
    EXPECT_EQ(aggrProto.last(), ui64(value.Last));
    EXPECT_EQ(aggrProto.count(), aggr.Count);
    EXPECT_EQ(aggrProto.mask(), aggr.Mask);
}

TEST(TAggregatesTest, Int64FromProto) {
    Aggregate proto;
    auto* protoAggr = proto.mutable_int64();
    protoAggr->set_max(10);
    protoAggr->set_min(20);
    protoAggr->set_sum(30);
    protoAggr->set_avg(40);
    protoAggr->set_last(50);
    protoAggr->set_count(60);
    protoAggr->set_mask(0x00111111);

    TAggregateVariant aggr;
    FromProto(proto, &aggr);

    ASSERT_TRUE(std::holds_alternative<TAggregateInt64>(aggr.Value));
    auto& value = std::get<TAggregateInt64>(aggr.Value);

    EXPECT_EQ(protoAggr->max(), ui64(value.Max));
    EXPECT_EQ(protoAggr->min(), ui64(value.Min));
    EXPECT_EQ(protoAggr->sum(), ui64(value.Sum));
    EXPECT_EQ(protoAggr->avg(), ui64(value.Avg));
    EXPECT_EQ(protoAggr->last(), ui64(value.Last));
    EXPECT_EQ(protoAggr->count(), aggr.Count);
    EXPECT_EQ(protoAggr->mask(), aggr.Mask);
}

TEST(TAggregatesTest, Int64Convert) {
    TAggregateVariant aggr1{
        .Value = TAggregateInt64{
            .Max = 1,
            .Min = 2,
            .Sum = 3,
            .Avg = 4,
            .Last = 5,
        },
        .Count = 6,
        .Mask = 0b01111110,
    };

    Aggregate proto;
    ToProto(aggr1, &proto);

    TAggregateVariant aggr2;
    FromProto(proto, &aggr2);

    EXPECT_EQ(aggr1, aggr2);
}

TEST(TAggregatesTest, SummaryDoubleToProto) {
    TAggregateVariant aggr;
    auto& value = aggr.Value.emplace<TAggregateSummaryDouble>();
    value.Last = NTs::NValue::TSummaryDouble{
        .CountValue = 1,
        .Sum = 2,
        .Min = 3,
        .Max = 4,
        .Last = 5,
    };
    value.Sum = NTs::NValue::TSummaryDouble{
        .CountValue = 10,
        .Sum = 20,
        .Min = 30,
        .Max = 40,
        .Last = 50,
    };
    aggr.Count = 37;
    aggr.Mask = 0b00101000;

    Aggregate proto;
    ToProto(aggr, &proto);

    ASSERT_EQ(proto.type_case(), Aggregate::kSummaryDouble);

    {
        auto& last = proto.summary_double().last();
        EXPECT_EQ(last.count(), ui64(value.Last.CountValue));
        EXPECT_EQ(last.sum(), value.Last.Sum);
        EXPECT_EQ(last.min(), value.Last.Min);
        EXPECT_EQ(last.max(), value.Last.Max);
        EXPECT_EQ(last.last(), value.Last.Last);
    }
    {
        auto& sum = proto.summary_double().sum();
        EXPECT_EQ(sum.count(), ui64(value.Sum.CountValue));
        EXPECT_EQ(sum.sum(), value.Sum.Sum);
        EXPECT_EQ(sum.min(), value.Sum.Min);
        EXPECT_EQ(sum.max(), value.Sum.Max);
        EXPECT_EQ(sum.last(), value.Sum.Last);
    }

    EXPECT_EQ(proto.summary_double().count(), aggr.Count);
    EXPECT_EQ(proto.summary_double().mask(), aggr.Mask);
}

TEST(TAggregatesTest, SummaryDoubleFromProto) {
    Aggregate proto;
    auto* protoSummary = proto.mutable_summary_double();
    protoSummary->set_count(73);
    protoSummary->set_mask(0b00101000);

    auto* lastProto = protoSummary->mutable_last();
    lastProto->set_count(1);
    lastProto->set_sum(2);
    lastProto->set_min(3);
    lastProto->set_max(4);
    lastProto->set_last(5);

    auto* sumProto = protoSummary->mutable_sum();
    sumProto->set_count(10);
    sumProto->set_sum(20);
    sumProto->set_min(30);
    sumProto->set_max(40);
    sumProto->set_last(50);

    TAggregateVariant aggr;
    FromProto(proto, &aggr);

    ASSERT_TRUE(std::holds_alternative<TAggregateSummaryDouble>(aggr.Value));
    auto& value = std::get<TAggregateSummaryDouble>(aggr.Value);

    EXPECT_EQ(lastProto->count(), ui64(value.Last.CountValue));
    EXPECT_EQ(lastProto->sum(), value.Last.Sum);
    EXPECT_EQ(lastProto->min(), value.Last.Min);
    EXPECT_EQ(lastProto->max(), value.Last.Max);
    EXPECT_EQ(lastProto->last(), value.Last.Last);

    EXPECT_EQ(sumProto->count(), ui64(value.Sum.CountValue));
    EXPECT_EQ(sumProto->sum(), value.Sum.Sum);
    EXPECT_EQ(sumProto->min(), value.Sum.Min);
    EXPECT_EQ(sumProto->max(), value.Sum.Max);
    EXPECT_EQ(sumProto->last(), value.Sum.Last);

    EXPECT_EQ(protoSummary->count(), aggr.Count);
    EXPECT_EQ(protoSummary->mask(), aggr.Mask);
}

TEST(TAggregatesTest, SummaryDoubleConvert) {
    TAggregateVariant aggr1{
        .Value = TAggregateSummaryDouble{
            .Last = {
                .CountValue = 1,
                .Sum = 2,
                .Min = 3,
                .Max = 4,
                .Last = 5,
            },
            .Sum = {
                .CountValue = 10,
                .Sum = 20,
                .Min = 30,
                .Max = 40,
                .Last = 50,
            },
        },
        .Count = 6,
        .Mask = 0b01111110,
    };

    Aggregate proto;
    ToProto(aggr1, &proto);

    TAggregateVariant aggr2;
    FromProto(proto, &aggr2);

    EXPECT_EQ(aggr1, aggr2);
}

TEST(TAggregatesTest, SummaryInt64ToProto) {
    TAggregateVariant aggr;
    auto& value = aggr.Value.emplace<TAggregateSummaryInt64>();
    value.Last = NTs::NValue::TSummaryInt{
        .CountValue = 1,
        .Sum = 2,
        .Min = 3,
        .Max = 4,
        .Last = 5,
    };
    value.Sum = NTs::NValue::TSummaryInt{
        .CountValue = 10,
        .Sum = 20,
        .Min = 30,
        .Max = 40,
        .Last = 50,
    };
    aggr.Count = 37;
    aggr.Mask = 0b00101000;

    Aggregate proto;
    ToProto(aggr, &proto);

    ASSERT_EQ(proto.type_case(), Aggregate::kSummaryInt64);

    {
        auto& last = proto.summary_int64().last();
        EXPECT_EQ(last.count(), ui64(value.Last.CountValue));
        EXPECT_EQ(last.sum(), value.Last.Sum);
        EXPECT_EQ(last.min(), value.Last.Min);
        EXPECT_EQ(last.max(), value.Last.Max);
        EXPECT_EQ(last.last(), value.Last.Last);
    }
    {
        auto& sum = proto.summary_int64().sum();
        EXPECT_EQ(sum.count(), ui64(value.Sum.CountValue));
        EXPECT_EQ(sum.sum(), value.Sum.Sum);
        EXPECT_EQ(sum.min(), value.Sum.Min);
        EXPECT_EQ(sum.max(), value.Sum.Max);
        EXPECT_EQ(sum.last(), value.Sum.Last);
    }

    EXPECT_EQ(proto.summary_int64().count(), aggr.Count);
    EXPECT_EQ(proto.summary_int64().mask(), aggr.Mask);
}

TEST(TAggregatesTest, SummaryInt64FromProto) {
    Aggregate proto;
    auto* protoSummary = proto.mutable_summary_int64();
    protoSummary->set_count(73);
    protoSummary->set_mask(0b00101000);

    auto* lastProto = protoSummary->mutable_last();
    lastProto->set_count(1);
    lastProto->set_sum(2);
    lastProto->set_min(3);
    lastProto->set_max(4);
    lastProto->set_last(5);

    auto* sumProto = protoSummary->mutable_sum();
    sumProto->set_count(10);
    sumProto->set_sum(20);
    sumProto->set_min(30);
    sumProto->set_max(40);
    sumProto->set_last(50);

    TAggregateVariant aggr;
    FromProto(proto, &aggr);

    ASSERT_TRUE(std::holds_alternative<TAggregateSummaryInt64>(aggr.Value));
    auto& value = std::get<TAggregateSummaryInt64>(aggr.Value);

    EXPECT_EQ(lastProto->count(), ui64(value.Last.CountValue));
    EXPECT_EQ(lastProto->sum(), value.Last.Sum);
    EXPECT_EQ(lastProto->min(), value.Last.Min);
    EXPECT_EQ(lastProto->max(), value.Last.Max);
    EXPECT_EQ(lastProto->last(), value.Last.Last);

    EXPECT_EQ(sumProto->count(), ui64(value.Sum.CountValue));
    EXPECT_EQ(sumProto->sum(), value.Sum.Sum);
    EXPECT_EQ(sumProto->min(), value.Sum.Min);
    EXPECT_EQ(sumProto->max(), value.Sum.Max);
    EXPECT_EQ(sumProto->last(), value.Sum.Last);

    EXPECT_EQ(protoSummary->count(), aggr.Count);
    EXPECT_EQ(protoSummary->mask(), aggr.Mask);
}

TEST(TAggregatesTest, SummaryInt64Convert) {
    TAggregateVariant aggr1{
        .Value = TAggregateSummaryInt64{
            .Last = {
                .CountValue = 1,
                .Sum = 2,
                .Min = 3,
                .Max = 4,
                .Last = 5,
            },
            .Sum = {
                .CountValue = 10,
                .Sum = 20,
                .Min = 30,
                .Max = 40,
                .Last = 50,
            },
        },
        .Count = 6,
        .Mask = 0b01111110,
    };

    Aggregate proto;
    ToProto(aggr1, &proto);

    TAggregateVariant aggr2;
    FromProto(proto, &aggr2);

    EXPECT_EQ(aggr1, aggr2);
}

TEST(TAggregatesTest, LogHistToProto) {
    TAggregateVariant aggr;
    auto& value = aggr.Value.emplace<TAggregateLogHistogram>();
    value.Last = NTs::NValue::TLogHistogram{
        .Values = {1.1, 2.2, 3.3, 4.4, 5.5},
        .ZeroCount = 6,
        .StartPower = 7,
        .MaxBucketCount = 8,
        .Base = 9.9,
    };
    value.Sum = NTs::NValue::TLogHistogram{
        .Values = {10.1, 20.2, 30.3, 40.4, 50.5},
        .ZeroCount = 60,
        .StartPower = 70,
        .MaxBucketCount = 80,
        .Base = 90.9,
    };
    aggr.Count = 77;
    aggr.Mask = 0b00101000;

    Aggregate proto;
    ToProto(aggr, &proto);

    ASSERT_EQ(proto.type_case(), Aggregate::kLogHistogram);

    {
        auto& last = proto.log_histogram().last();
        std::vector<double> buckets(last.buckets().begin(), last.buckets().end());
        EXPECT_EQ(buckets, value.Last.Values);
        EXPECT_EQ(last.zeroes(), value.Last.ZeroCount);
        EXPECT_EQ(last.start_power(), value.Last.StartPower);
        EXPECT_EQ(last.max_buckets_size(), value.Last.MaxBucketCount);
        EXPECT_EQ(last.base(), value.Last.Base);
    }
    {
        auto& sum = proto.log_histogram().sum();
        std::vector<double> buckets(sum.buckets().begin(), sum.buckets().end());
        EXPECT_EQ(buckets, value.Sum.Values);
        EXPECT_EQ(sum.zeroes(), value.Sum.ZeroCount);
        EXPECT_EQ(sum.start_power(), value.Sum.StartPower);
        EXPECT_EQ(sum.max_buckets_size(), value.Sum.MaxBucketCount);
        EXPECT_EQ(sum.base(), value.Sum.Base);
    }

    EXPECT_EQ(proto.log_histogram().count(), aggr.Count);
    EXPECT_EQ(proto.log_histogram().mask(), aggr.Mask);
}

TEST(TAggregatesTest, LogHistFromProto) {
    Aggregate proto;
    auto* protoSummary = proto.mutable_log_histogram();
    protoSummary->set_count(77);
    protoSummary->set_mask(0b00101000);

    auto* lastProto = protoSummary->mutable_last();
    lastProto->add_buckets(1.1);
    lastProto->add_buckets(2.2);
    lastProto->add_buckets(3.3);
    lastProto->set_zeroes(4);
    lastProto->set_start_power(5);
    lastProto->set_max_buckets_size(6);
    lastProto->set_base(7.7);

    auto* sumProto = protoSummary->mutable_sum();
    sumProto->add_buckets(10.1);
    sumProto->add_buckets(20.2);
    sumProto->add_buckets(30.3);
    sumProto->set_zeroes(40);
    sumProto->set_start_power(50);
    sumProto->set_max_buckets_size(60);
    sumProto->set_base(70.7);

    TAggregateVariant aggr;
    FromProto(proto, &aggr);

    ASSERT_TRUE(std::holds_alternative<TAggregateLogHistogram>(aggr.Value));
    auto& value = std::get<TAggregateLogHistogram>(aggr.Value);

    {
        std::vector<double> buckets(lastProto->buckets().begin(), lastProto->buckets().end());
        EXPECT_EQ(buckets, value.Last.Values);
        EXPECT_EQ(lastProto->zeroes(), value.Last.ZeroCount);
        EXPECT_EQ(lastProto->start_power(), value.Last.StartPower);
        EXPECT_EQ(lastProto->max_buckets_size(), value.Last.MaxBucketCount);
        EXPECT_EQ(lastProto->base(), value.Last.Base);
    }
    {
        std::vector<double> buckets(sumProto->buckets().begin(), sumProto->buckets().end());
        EXPECT_EQ(buckets, value.Sum.Values);
        EXPECT_EQ(sumProto->zeroes(), value.Sum.ZeroCount);
        EXPECT_EQ(sumProto->start_power(), value.Sum.StartPower);
        EXPECT_EQ(sumProto->max_buckets_size(), value.Sum.MaxBucketCount);
        EXPECT_EQ(sumProto->base(), value.Sum.Base);
    }

    EXPECT_EQ(protoSummary->count(), aggr.Count);
    EXPECT_EQ(protoSummary->mask(), aggr.Mask);
}

TEST(TAggregatesTest, LogHistConvert) {
    TAggregateVariant aggr1{
        .Value = TAggregateLogHistogram{
            .Last = {
                .Values = {1.1, 2.2, 3.3, 4.4, 5.5},
                .ZeroCount = 6,
                .StartPower = 7,
                .MaxBucketCount = 8,
                .Base = 9.9,
            },
            .Sum = {
                .Values = {10.1, 20.2, 30.3, 40.4, 50.5},
                .ZeroCount = 60,
                .StartPower = 70,
                .MaxBucketCount = 80,
                .Base = 90.9,
            },
        },
        .Count = 6,
        .Mask = 0b01111110,
    };

    Aggregate proto;
    ToProto(aggr1, &proto);

    TAggregateVariant aggr2;
    FromProto(proto, &aggr2);

    EXPECT_EQ(aggr1, aggr2);
}

TEST(TAggregatesTest, HistToProto) {
    TAggregateVariant aggr;
    auto& value = aggr.Value.emplace<TAggregateHistogram>();
    value.Last = NTs::NValue::THistogram{
        .Denom = 1000,
        .Buckets = {
            {1.1, 2},
            {3.3, 4},
            {5.5, 6},
        },
    };
    value.Sum = NTs::NValue::THistogram{
        .Denom = 15'000,
        .Buckets = {
            {10.1, 20},
            {30.3, 40},
            {50.5, 60},
        },
    };
    aggr.Count = 77;
    aggr.Mask = 0b00101000;

    Aggregate proto;
    ToProto(aggr, &proto);

    ASSERT_EQ(proto.type_case(), Aggregate::kHistogram);

    {
        auto& last = proto.histogram().last();
        ASSERT_EQ(last.buckets_size(), last.bounds_size());
        ASSERT_EQ(size_t(last.buckets_size()), value.Last.Buckets.size());

        for (int i = 0; i < last.buckets_size(); i++) {
            ASSERT_EQ(last.bounds(i), value.Last.Buckets[i].UpperBound) << "i=" << i;
            ASSERT_EQ(last.buckets(i), value.Last.Buckets[i].Value) << "i=" << i;
        }

        EXPECT_EQ(last.denom(), value.Last.Denom);
    }
    {
        auto& sum = proto.histogram().sum();
        ASSERT_EQ(sum.buckets_size(), sum.bounds_size());
        ASSERT_EQ(size_t(sum.buckets_size()), value.Sum.Buckets.size());

        for (int i = 0; i < sum.buckets_size(); i++) {
            ASSERT_EQ(sum.bounds(i), value.Sum.Buckets[i].UpperBound) << "i=" << i;
            ASSERT_EQ(sum.buckets(i), value.Sum.Buckets[i].Value) << "i=" << i;
        }

        EXPECT_EQ(sum.denom(), value.Sum.Denom);
    }

    EXPECT_EQ(proto.histogram().count(), aggr.Count);
    EXPECT_EQ(proto.histogram().mask(), aggr.Mask);
}

TEST(TAggregatesTest, HistFromProto) {
    Aggregate proto;
    auto* protoSummary = proto.mutable_histogram();
    protoSummary->set_count(77);
    protoSummary->set_mask(0b00101000);

    auto* lastProto = protoSummary->mutable_last();
    lastProto->add_bounds(1.1); lastProto->add_buckets(2);
    lastProto->add_bounds(3.3); lastProto->add_buckets(4);
    lastProto->add_bounds(5.5); lastProto->add_buckets(6);
    lastProto->set_denom(7);

    auto* sumProto = protoSummary->mutable_sum();
    sumProto->add_bounds(10.1); sumProto->add_buckets(20);
    sumProto->add_bounds(30.3); sumProto->add_buckets(40);
    sumProto->add_bounds(50.5); sumProto->add_buckets(60);
    sumProto->set_denom(70);

    TAggregateVariant aggr;
    FromProto(proto, &aggr);

    ASSERT_TRUE(std::holds_alternative<TAggregateHistogram>(aggr.Value));
    auto& value = std::get<TAggregateHistogram>(aggr.Value);

    {
        ASSERT_EQ(lastProto->buckets_size(), lastProto->buckets_size());
        ASSERT_EQ(size_t(lastProto->buckets_size()), value.Last.Buckets.size());

        for (int i = 0; i < lastProto->buckets_size(); i++) {
            ASSERT_EQ(lastProto->bounds(i), value.Last.Buckets[i].UpperBound) << "i=" << i;
            ASSERT_EQ(lastProto->buckets(i), value.Last.Buckets[i].Value) << "i=" << i;
        }

        EXPECT_EQ(lastProto->denom(), value.Last.Denom);
    }
    {
        ASSERT_EQ(sumProto->buckets_size(), sumProto->buckets_size());
        ASSERT_EQ(size_t(sumProto->buckets_size()), value.Sum.Buckets.size());

        for (int i = 0; i < sumProto->buckets_size(); i++) {
            ASSERT_EQ(sumProto->bounds(i), value.Sum.Buckets[i].UpperBound) << "i=" << i;
            ASSERT_EQ(sumProto->buckets(i), value.Sum.Buckets[i].Value) << "i=" << i;
        }

        EXPECT_EQ(sumProto->denom(), value.Sum.Denom);
    }

    EXPECT_EQ(protoSummary->count(), aggr.Count);
    EXPECT_EQ(protoSummary->mask(), aggr.Mask);
}

TEST(TAggregatesTest, HistConvert) {
    TAggregateVariant aggr1{
        .Value = TAggregateHistogram{
            .Last = {
                .Denom = 1000,
                .Buckets = {
                    {1.1, 2},
                    {3.3, 4},
                    {5.5, 6},
                },
            },
            .Sum = {
                .Denom = 15'000,
                .Buckets = {
                    {10.1, 20},
                    {30.3, 40},
                    {50.5, 60},
                },
            },
        },
        .Count = 6,
        .Mask = 0b01111110,
    };

    Aggregate proto;
    ToProto(aggr1, &proto);

    TAggregateVariant aggr2;
    FromProto(proto, &aggr2);

    EXPECT_EQ(aggr1, aggr2);
}

TEST(TAggregationExtractorTest, PositiveAggregationTest) {
    yandex::solomon::stockpile::MetricData metricData;
    auto doubleAggregate = metricData.mutable_double_();
    double min = 1, max = 2, sum = 3, avg = 4, last = 5, count = 6;
    doubleAggregate->set_min(min);
    doubleAggregate->set_max(max);
    doubleAggregate->set_sum(sum);
    doubleAggregate->set_avg(avg);
    doubleAggregate->set_last(last);
    doubleAggregate->set_count(count);
    {
        yandex::solomon::math::Aggregation aggregation = yandex::solomon::math::Aggregation::MIN;
        auto extractor = GetSummaryFieldExtractor<double>(aggregation);
        std::optional<double> extractedValue = extractor(&metricData);
        EXPECT_TRUE(extractedValue);
        EXPECT_EQ(extractedValue.value_or(0), min);
    }
    {
        yandex::solomon::math::Aggregation aggregation = yandex::solomon::math::Aggregation::MAX;
        auto extractor = GetSummaryFieldExtractor<double>(aggregation);
        std::optional<double> extractedValue = extractor(&metricData);
        EXPECT_TRUE(extractedValue);
        EXPECT_EQ(extractedValue.value_or(0), max);
    }
    {
        yandex::solomon::math::Aggregation aggregation = yandex::solomon::math::Aggregation::SUM;
        auto extractor = GetSummaryFieldExtractor<double>(aggregation);
        std::optional<double> extractedValue = extractor(&metricData);
        EXPECT_TRUE(extractedValue);
        EXPECT_EQ(extractedValue.value_or(0), sum);
    }
    {
        yandex::solomon::math::Aggregation aggregation = yandex::solomon::math::Aggregation::AVG;
        auto extractor = GetSummaryFieldExtractor<double>(aggregation);
        std::optional<double> extractedValue = extractor(&metricData);
        EXPECT_TRUE(extractedValue);
        EXPECT_EQ(extractedValue.value_or(0), avg);
    }
    {
        yandex::solomon::math::Aggregation aggregation = yandex::solomon::math::Aggregation::LAST;
        auto extractor = GetSummaryFieldExtractor<double>(aggregation);
        std::optional<double> extractedValue = extractor(&metricData);
        EXPECT_TRUE(extractedValue);
        EXPECT_EQ(extractedValue.value_or(0), last);
    }
    {
        yandex::solomon::math::Aggregation aggregation = yandex::solomon::math::Aggregation::COUNT;
        auto extractor = GetSummaryFieldExtractor<double>(aggregation);
        std::optional<double> extractedValue = extractor(&metricData);
        EXPECT_TRUE(extractedValue);
        EXPECT_EQ(extractedValue.value_or(0), count);
    }
    {
        yandex::solomon::math::Aggregation aggregation = yandex::solomon::math::Aggregation::DEFAULT_AGGREGATION;
        auto extractor = GetSummaryFieldExtractor<double>(aggregation);
        std::optional<double> extractedValue = extractor(&metricData);
        EXPECT_TRUE(extractedValue);
        EXPECT_EQ(extractedValue.value_or(0), last);
    }
}

TEST(TAggregationExtractorTest, NegativeAggregationTest) {
    yandex::solomon::stockpile::MetricData metricData;
    yandex::solomon::math::Aggregation aggregation = yandex::solomon::math::Aggregation::MIN;
    auto extractor = GetSummaryFieldExtractor<double>(aggregation);
    EXPECT_FALSE(extractor(&metricData));
}
