#include "points.h"

#include <solomon/services/dataproxy/lib/timeseries/compression.h>
#include <solomon/services/dataproxy/lib/timeseries/empty.h>
#include <solomon/services/dataproxy/lib/timeseries/vector.h>

#include <library/cpp/testing/gtest/gtest.h>

#include <util/generic/buffer.h>

using namespace NSolomon;
using namespace NDataProxy;
using namespace NMonitoring;

TEST(TCompressionTest, CompressDecompress) {
    TVectorTimeSeries<NTs::TLongPoint> timeSeries{EMetricType::COUNTER, {
        LongPoint("2020-01-02T03:04:05Z", 1),
        LongPoint("2020-01-02T03:04:10Z", 2),
        LongPoint("2020-01-02T03:04:15Z", 3),
        LongPoint("2020-01-02T03:04:20Z", 4),
    }};

    auto [buffer, pointCount] = Compress(timeSeries);
    EXPECT_NE(buffer.Size(), size_t(0));
    EXPECT_EQ(pointCount, ui32(4));

    NTs::TVariantPoint point;
    auto it = Decompress(EMetricType::COUNTER, NTs::TLongPoint::SimpleColumns, buffer);
    ASSERT_TRUE(it);

    ASSERT_TRUE(it->Next(&point));
    EXPECT_EQ(point.Time, timeSeries[0].Time);
    EXPECT_EQ(point.Get<NTs::NValue::TLong>().Value, timeSeries[0].Value);

    ASSERT_TRUE(it->Next(&point));
    EXPECT_EQ(point.Time, timeSeries[1].Time);
    EXPECT_EQ(point.Get<NTs::NValue::TLong>().Value, timeSeries[1].Value);

    ASSERT_TRUE(it->Next(&point));
    EXPECT_EQ(point.Time, timeSeries[2].Time);
    EXPECT_EQ(point.Get<NTs::NValue::TLong>().Value, timeSeries[2].Value);

    ASSERT_TRUE(it->Next(&point));
    EXPECT_EQ(point.Time, timeSeries[3].Time);
    EXPECT_EQ(point.Get<NTs::NValue::TLong>().Value, timeSeries[3].Value);

    ASSERT_FALSE(it->Next(&point));
}

TEST(TCompressionTest, EmptyUnknownTimeSeries) {
    TEmptyTimeSeries timeSeries{EMetricType::UNKNOWN};
    auto [buffer, pointCount] = Compress(timeSeries);
    EXPECT_EQ(buffer.Size(), size_t(0));
    EXPECT_EQ(pointCount, ui32(0));

    NTs::TVariantPoint point;
    auto it = Decompress(NMonitoring::EMetricType::UNKNOWN, NTs::TColumnSet{}, buffer);
    ASSERT_FALSE(it->Next(&point));
}

TEST(TCompressionTest, NonEmptyUnknownTimeSeries) {
    struct TEndlessIter: public ITimeSeriesIter {
        bool Next(NTs::TVariantPoint* /*point*/) override {
            // imitate endless time series
            return true;
        }
    };

    ASSERT_THROW_MESSAGE_HAS_SUBSTR({
        TEndlessIter iter;
        Compress(EMetricType::UNKNOWN, NTs::TColumnSet{}, iter);
    }, yexception, "cannot compress non empty time series with unknown type");
}
