#include "points.h"

#include <solomon/services/dataproxy/lib/timeseries/merger.h>
#include <solomon/services/dataproxy/lib/timeseries/vector.h>

#include <library/cpp/testing/gtest/gtest.h>

using namespace NSolomon;
using namespace NDataProxy;
using namespace NMonitoring;

TEST(TMergeTimeSeriesTest, MaxMerger) {
    TVectorTimeSeries<NTs::TDoublePoint> ts1{EMetricType::GAUGE, {
        DoublePoint("2020-12-20T11:12:00Z", 1),
        // 2020-12-20T11:12:15Z  <gap>
        DoublePoint("2020-12-20T11:12:30Z", 3),
        // 2020-12-20T11:12:45Z  <gap>
        DoublePoint("2020-12-20T11:13:00Z", 5),
    }};

    TVectorTimeSeries<NTs::TDoublePoint> ts2{EMetricType::GAUGE, {
        // 2020-12-20T11:12:00Z  <gap>
        DoublePoint("2020-12-20T11:12:15Z", 2),
        DoublePoint("2020-12-20T11:12:30Z", 3),
        DoublePoint("2020-12-20T11:12:45Z", 4),
        // 2020-12-20T11:13:00Z gap
    }};

    TMergingIterator<TMaxMerger> it;
    it.AddTimeSeries(ts1);
    it.AddTimeSeries(ts2);

    NTs::TVariantPoint point;

    ASSERT_TRUE(it.Next(&point));
    EXPECT_EQ(point.Time, TInstant::ParseIso8601("2020-12-20T11:12:00Z"));
    EXPECT_EQ(point.Get<NTs::NValue::TDouble>().ValueDivided(), 1.0);

    ASSERT_TRUE(it.Next(&point));
    EXPECT_EQ(point.Time, TInstant::ParseIso8601("2020-12-20T11:12:15Z"));
    EXPECT_EQ(point.Get<NTs::NValue::TDouble>().ValueDivided(), 2.0);

    ASSERT_TRUE(it.Next(&point));
    EXPECT_EQ(point.Time, TInstant::ParseIso8601("2020-12-20T11:12:30Z"));
    EXPECT_EQ(point.Get<NTs::NValue::TDouble>().ValueDivided(), 3.0);

    ASSERT_TRUE(it.Next(&point));
    EXPECT_EQ(point.Time, TInstant::ParseIso8601("2020-12-20T11:12:45Z"));
    EXPECT_EQ(point.Get<NTs::NValue::TDouble>().ValueDivided(), 4.0);

    ASSERT_TRUE(it.Next(&point));
    EXPECT_EQ(point.Time, TInstant::ParseIso8601("2020-12-20T11:13:00Z"));
    EXPECT_EQ(point.Get<NTs::NValue::TDouble>().ValueDivided(), 5.0);

    ASSERT_FALSE(it.Next(&point));
}

TEST(TMergeTimeSeriesTest, CombineMerger) {
    TVectorTimeSeries<NTs::TLongPoint> ts1{EMetricType::IGAUGE, {
        LongPoint("2020-12-20T11:12:00Z", 1, TDuration::Seconds(15), 3),
        LongPoint("2020-12-20T11:12:15Z", 2, TDuration::Seconds(15), 3),
        LongPoint("2020-12-20T11:12:30Z", 3, TDuration::Seconds(15), 3),
    }};

    TVectorTimeSeries<NTs::TLongPoint> ts2{EMetricType::IGAUGE, {
        LongPoint("2020-12-20T11:12:00Z", 5, TDuration::Seconds(15), 5),
        LongPoint("2020-12-20T11:12:15Z", 4, TDuration::Seconds(15), 5),
        LongPoint("2020-12-20T11:12:30Z", 3, TDuration::Seconds(15), 5),
    }};

    TMergingIterator<TCombineMerger> it;
    it.AddTimeSeries(ts1);
    it.AddTimeSeries(ts2);

    NTs::TVariantPoint point;

    ASSERT_TRUE(it.Next(&point));
    EXPECT_EQ(point.Time, TInstant::ParseIso8601("2020-12-20T11:12:00Z"));
    EXPECT_EQ(point.Step, TDuration::Seconds(15));
    EXPECT_EQ(point.Count, ui64(3 + 5));
    EXPECT_EQ(point.Get<NTs::NValue::TLong>().Value, 1 + 5);

    ASSERT_TRUE(it.Next(&point));
    EXPECT_EQ(point.Time, TInstant::ParseIso8601("2020-12-20T11:12:15Z"));
    EXPECT_EQ(point.Step, TDuration::Seconds(15));
    EXPECT_EQ(point.Count, ui64(3 + 5));
    EXPECT_EQ(point.Get<NTs::NValue::TLong>().Value, 2 + 4);

    ASSERT_TRUE(it.Next(&point));
    EXPECT_EQ(point.Time, TInstant::ParseIso8601("2020-12-20T11:12:30Z"));
    EXPECT_EQ(point.Step, TDuration::Seconds(15));
    EXPECT_EQ(point.Count, ui64(3 + 5));
    EXPECT_EQ(point.Get<NTs::NValue::TLong>().Value, 3 + 3);

    ASSERT_FALSE(it.Next(&point));
}
