#include "points.h"

#include <solomon/services/dataproxy/lib/timeseries/protobuf.h>
#include <solomon/services/dataproxy/lib/timeseries/vector.h>

#include <solomon/libs/cpp/stockpile_codec/format.h>

#include <library/cpp/testing/gtest/gtest.h>

using namespace NSolomon;
using namespace NDataProxy;
using namespace NMonitoring;

class TProtobufTimeSeriesTest: public ::testing::TestWithParam<ui32> {
protected:
    TProtobufTimeSeriesTest()
        : FormatNumber_(GetParam())
        , Format_(NStockpile::FormatFromInt(FormatNumber_))
    {
    }
    ui32 FormatNumber_;
    NStockpile::EFormat Format_;
};

INSTANTIATE_TEST_SUITE_P(PerFormatVersion, TProtobufTimeSeriesTest,
        ::testing::Range(static_cast<ui32>(NStockpile::MinSupportedFormat), static_cast<ui32>(NStockpile::MaxSupportedFormat) + 1),
        ::testing::PrintToStringParamName());

TEST_P(TProtobufTimeSeriesTest, NotCompressed) {
    TVectorTimeSeries<NTs::TLongPoint> timeSeries{EMetricType::RATE, {
        LongPoint("2020-01-02T03:04:05Z", 1),
        LongPoint("2020-01-02T03:04:10Z", 2),
        LongPoint("2020-01-02T03:04:15Z", 3),
    }};

    yandex::solomon::model::TimeSeries proto;
    ToProto(timeSeries, &proto, Format_);

    EXPECT_EQ(proto.format_version(), FormatNumber_);
    EXPECT_EQ(proto.chunks_size(), 1);

    const auto& chunk = proto.chunks(0);
    EXPECT_EQ(chunk.point_count(), ui32(3));
    EXPECT_TRUE(!chunk.content().empty());

    // TODO: fill
    EXPECT_EQ(chunk.from_millis(), ui32(0));
    EXPECT_EQ(chunk.to_millis(), ui32(0));
}

TEST_P(TProtobufTimeSeriesTest, AlreadyCompressed) {
    TVectorTimeSeries<NTs::TDoublePoint> vecTs{EMetricType::GAUGE, {
        DoublePoint("2020-01-02T03:04:05Z", 1),
        DoublePoint("2020-01-02T03:04:10Z", 2),
        DoublePoint("2020-01-02T03:04:15Z", 3),
    }};
    TCompressedTimeSeries cmpTs{vecTs};

    yandex::solomon::model::TimeSeries proto;
    ToProto(cmpTs, &proto, Format_);

    EXPECT_EQ(proto.format_version(), FormatNumber_);
    EXPECT_EQ(proto.chunks_size(), 1);

    const auto& chunk = proto.chunks(0);
    EXPECT_EQ(chunk.point_count(), ui32(3));
    EXPECT_TRUE(!chunk.content().empty());

    // TODO: fill
    EXPECT_EQ(chunk.from_millis(), ui32(0));
    EXPECT_EQ(chunk.to_millis(), ui32(0));
}

TEST_P(TProtobufTimeSeriesTest, FromProto_NoChunks) {
    yandex::solomon::stockpile::TCompressedReadResponse resp;
    resp.SetBinaryVersion(static_cast<i32>(Format_));
    resp.SetType(yandex::solomon::model::MetricType::COUNTER);

    auto timeSeries = FromProto(EMetricType::RATE, resp);
    EXPECT_EQ(timeSeries->Type(), EMetricType::COUNTER);
    EXPECT_EQ(timeSeries->Columns(), NTs::TLongPoint::SimpleColumns);
    EXPECT_EQ(timeSeries->PointCount(), ui32(0));

    auto it = timeSeries->Iterator();
    ASSERT_FALSE(it->Next(nullptr));
}

template <typename TPoint>
void MakeChunk(
        yandex::solomon::model::TimeSeries_Chunk* chunk,
        EMetricType type,
        std::vector<TPoint> points,
        NStockpile::EFormat format)
{
    TVectorTimeSeries<TPoint> timeSeries{type, std::move(points)};
    yandex::solomon::model::TimeSeries proto;
    ToProto(timeSeries, &proto, format);

    chunk->set_point_count(proto.chunks(0).point_count());
    chunk->set_content(proto.chunks(0).content());
    chunk->set_from_millis(proto.chunks(0).from_millis());
    chunk->set_to_millis(proto.chunks(0).to_millis());
}

TEST_P(TProtobufTimeSeriesTest, FromProto_SingleChunks) {
    yandex::solomon::stockpile::TCompressedReadResponse resp;
    resp.SetBinaryVersion(static_cast<i32>(FormatNumber_));
    resp.SetType(yandex::solomon::model::MetricType::COUNTER);

    MakeChunk<NTs::TLongPoint>(resp.AddChunks(), EMetricType::COUNTER, {
        LongPoint("2020-01-02T03:04:05Z", 10),
        LongPoint("2020-01-02T03:04:20Z", 20),
        LongPoint("2020-01-02T03:04:35Z", 30),
    }, Format_);

    auto timeSeries = FromProto(EMetricType::RATE, resp);
    EXPECT_EQ(timeSeries->Type(), EMetricType::COUNTER);
    EXPECT_EQ(timeSeries->Columns(), NTs::TLongPoint::SimpleColumns);
    EXPECT_EQ(timeSeries->PointCount(), ui32(3));

    auto it = timeSeries->Iterator();
    NTs::TVariantPoint point;

    ASSERT_TRUE(it->Next(&point));
    EXPECT_EQ(point.Time, TInstant::ParseIso8601("2020-01-02T03:04:05Z"));
    EXPECT_EQ(point.Get<NTs::NValue::TLong>().Value, 10);

    ASSERT_TRUE(it->Next(&point));
    EXPECT_EQ(point.Time, TInstant::ParseIso8601("2020-01-02T03:04:20Z"));
    EXPECT_EQ(point.Get<NTs::NValue::TLong>().Value, 20);

    ASSERT_TRUE(it->Next(&point));
    EXPECT_EQ(point.Time, TInstant::ParseIso8601("2020-01-02T03:04:35Z"));
    EXPECT_EQ(point.Get<NTs::NValue::TLong>().Value, 30);

    ASSERT_FALSE(it->Next(&point));
}

TEST_P(TProtobufTimeSeriesTest, FromProto_MultipleChunks) {
    yandex::solomon::stockpile::TCompressedReadResponse resp;
    resp.SetBinaryVersion(static_cast<ui32>(NStockpile::MaxSupportedFormat));
    resp.SetType(yandex::solomon::model::MetricType::COUNTER);

    MakeChunk<NTs::TLongPoint>(resp.AddChunks(), EMetricType::COUNTER, {
        LongPoint("2020-01-02T03:04:05Z", 10),
        LongPoint("2020-01-02T03:04:20Z", 20),
    }, Format_);
    MakeChunk<NTs::TLongPoint>(resp.AddChunks(), EMetricType::COUNTER, {
        LongPoint("2020-01-02T03:04:35Z", 30),
        LongPoint("2020-01-02T03:04:50Z", 40),
    }, Format_);
    MakeChunk<NTs::TLongPoint>(resp.AddChunks(), EMetricType::COUNTER, {
        LongPoint("2020-01-02T03:05:05Z", 50),
    }, Format_);

    auto timeSeries = FromProto(EMetricType::RATE, resp);
    EXPECT_EQ(timeSeries->Type(), EMetricType::COUNTER);
    EXPECT_EQ(timeSeries->Columns(), NTs::TLongPoint::SimpleColumns);
    EXPECT_EQ(timeSeries->PointCount(), ui32(5));

    auto it = timeSeries->Iterator();
    NTs::TVariantPoint point;

    ASSERT_TRUE(it->Next(&point));
    EXPECT_EQ(point.Time, TInstant::ParseIso8601("2020-01-02T03:04:05Z"));
    EXPECT_EQ(point.Get<NTs::NValue::TLong>().Value, 10);

    ASSERT_TRUE(it->Next(&point));
    EXPECT_EQ(point.Time, TInstant::ParseIso8601("2020-01-02T03:04:20Z"));
    EXPECT_EQ(point.Get<NTs::NValue::TLong>().Value, 20);

    ASSERT_TRUE(it->Next(&point));
    EXPECT_EQ(point.Time, TInstant::ParseIso8601("2020-01-02T03:04:35Z"));
    EXPECT_EQ(point.Get<NTs::NValue::TLong>().Value, 30);

    ASSERT_TRUE(it->Next(&point));
    EXPECT_EQ(point.Time, TInstant::ParseIso8601("2020-01-02T03:04:50Z"));
    EXPECT_EQ(point.Get<NTs::NValue::TLong>().Value, 40);

    ASSERT_TRUE(it->Next(&point));
    EXPECT_EQ(point.Time, TInstant::ParseIso8601("2020-01-02T03:05:05Z"));
    EXPECT_EQ(point.Get<NTs::NValue::TLong>().Value, 50);

    ASSERT_FALSE(it->Next(&point));
}
