#include "points.h"

#include <solomon/services/dataproxy/lib/timeseries/vector.h>

#include <library/cpp/testing/gtest/gtest.h>

using namespace NSolomon;
using namespace NDataProxy;
using namespace NMonitoring;

TEST(TVectorTimeSeriesTest, Empty) {
    TVectorTimeSeries<NTs::TDoublePoint> timeSeries{EMetricType::GAUGE};

    EXPECT_EQ(timeSeries.Type(), EMetricType::GAUGE);
    EXPECT_EQ(timeSeries.Columns(), NTs::TDoublePoint::SimpleColumns);
    ASSERT_EQ(timeSeries.PointCount(), ui32(0));

    NTs::TVariantPoint point;
    auto it = timeSeries.Iterator();
    ASSERT_FALSE(it->Next(&point));
}

TEST(TVectorTimeSeriesTest, NonEmpty) {
    TVectorTimeSeries<NTs::TLongPoint> timeSeries{EMetricType::RATE, {
        LongPoint("2020-01-02T03:04:05Z", 1),
        LongPoint("2020-01-02T03:04:10Z", 2),
    }};

    EXPECT_EQ(timeSeries.Type(), EMetricType::RATE);
    EXPECT_EQ(timeSeries.Columns(), NTs::TLongPoint::SimpleColumns);
    ASSERT_EQ(timeSeries.PointCount(), ui32(2));

    EXPECT_EQ(timeSeries[0].Time, TInstant::ParseIso8601("2020-01-02T03:04:05Z"));
    EXPECT_EQ(timeSeries[0].Value, 1);

    EXPECT_EQ(timeSeries[1].Time, TInstant::ParseIso8601("2020-01-02T03:04:10Z"));
    EXPECT_EQ(timeSeries[1].Value, 2);

    NTs::TVariantPoint point;
    auto it = timeSeries.Iterator();

    ASSERT_TRUE(it->Next(&point));
    EXPECT_EQ(point.Time, TInstant::ParseIso8601("2020-01-02T03:04:05Z"));
    EXPECT_EQ(point.Get<NTs::NValue::TLong>().Value, 1);

    ASSERT_TRUE(it->Next(&point));
    EXPECT_EQ(point.Time, TInstant::ParseIso8601("2020-01-02T03:04:10Z"));
    EXPECT_EQ(point.Get<NTs::NValue::TLong>().Value, 2);

    ASSERT_FALSE(it->Next(&point));
}

TEST(TVectorTimeSeriesTest, PushBack) {
    TVectorTimeSeries<NTs::TSummaryDoublePoint> timeSeries{EMetricType::DSUMMARY};

    EXPECT_EQ(timeSeries.Type(), EMetricType::DSUMMARY);
    EXPECT_EQ(timeSeries.Columns(), NTs::TSummaryDoublePoint::SimpleColumns);
    ASSERT_EQ(timeSeries.PointCount(), ui32(0));

    timeSeries.PushBack(
        SummaryDoublePoint("2020-01-02T03:04:05Z", {
            .CountValue = 10,
            .Sum = 20,
            .Min = 1,
            .Max = 10,
            .Last = 5
        })
    );

    timeSeries.PushBack(
        SummaryDoublePoint("2020-01-02T03:04:10Z", {
            .CountValue = 5,
            .Sum = 30,
            .Min = 5,
            .Max = 20,
            .Last = 10
        })
    );

    NTs::TVariantPoint point;
    auto it = timeSeries.Iterator();

    {
        ASSERT_TRUE(it->Next(&point));
        EXPECT_EQ(point.Time, TInstant::ParseIso8601("2020-01-02T03:04:05Z"));

        auto& summary = point.Get<NTs::NValue::TSummaryDouble>();
        EXPECT_EQ(summary.CountValue, 10);
        EXPECT_EQ(summary.Sum, 20);
        EXPECT_EQ(summary.Min, 1);
        EXPECT_EQ(summary.Max, 10);
        EXPECT_EQ(summary.Last, 5);
    }
    {
        ASSERT_TRUE(it->Next(&point));
        EXPECT_EQ(point.Time, TInstant::ParseIso8601("2020-01-02T03:04:10Z"));

        auto& summary = point.Get<NTs::NValue::TSummaryDouble>();
        EXPECT_EQ(summary.CountValue, 5);
        EXPECT_EQ(summary.Sum, 30);
        EXPECT_EQ(summary.Min, 5);
        EXPECT_EQ(summary.Max, 20);
        EXPECT_EQ(summary.Last, 10);
    }

    ASSERT_FALSE(it->Next(&point));
}
