#pragma once

#include "timeseries.h"
#include "type_to_columns.h"

#include <vector>

namespace NSolomon::NDataProxy {

template <typename TPoint>
class TVectorTimeSeries final: public ITimeSeries {
    using TVecIt = typename std::vector<TPoint>::const_iterator;
private:
    struct TIter final: public ITimeSeriesIter {
        TVecIt It;
        TVecIt End;

        TIter(TVecIt it, TVecIt end)
            : It{it}
            , End{end}
        {
        }

        bool Next(NTs::TVariantPoint* point) override {
            if (It == End) {
                return false;
            }

            point->Time = It->Time;
            point->Step = It->Step;
            point->Count = It->Count;
            point->Merge = It->Merge;
            point->Set(*It);

            ++It;
            return true;
        }
    };

public:
    explicit TVectorTimeSeries(NMonitoring::EMetricType type)
        : Type_{type}
        , Columns_{TPoint::SimpleColumns}
    {
        EnsureValidTypeForColumns(Type_, Columns_);
    }

    TVectorTimeSeries(NMonitoring::EMetricType type, NTs::TColumnSet columns)
        : Type_{type}
        , Columns_{columns}
    {
        EnsureValidTypeForColumns(Type_, Columns_);
    }

    TVectorTimeSeries(NMonitoring::EMetricType type, std::vector<TPoint> points)
        : Type_{type}
        , Columns_{TPoint::SimpleColumns}
        , Points_{std::move(points)}
    {
        EnsureValidTypeForColumns(Type_, Columns_);
    }

    TVectorTimeSeries(NMonitoring::EMetricType type, NTs::TColumnSet columns, std::vector<TPoint> points)
        : Type_{type}
        , Columns_{columns}
        , Points_{std::move(points)}
    {
        EnsureValidTypeForColumns(Type_, Columns_);
    }

    NMonitoring::EMetricType Type() const override {
        return Type_;
    }

    NTs::TColumnSet Columns() const override {
        return Columns_;
    }

    ui32 PointCount() const override {
        return static_cast<ui32>(Points_.size());
    }

    std::unique_ptr<ITimeSeriesIter> Iterator() const override {
        return std::make_unique<TIter>(Points_.begin(), Points_.end());
    }

    const TPoint& operator[](size_t idx) const {
        return Points_[idx];
    }

    TPoint& operator[](size_t idx) {
        return Points_[idx];
    }

    void PushBack(const TPoint& point) {
        Points_.push_back(point);
    }

    void PushBack(TPoint&& point) {
        Points_.push_back(std::move(point));
    }

private:
    NMonitoring::EMetricType Type_;
    NTs::TColumnSet Columns_;
    std::vector<TPoint> Points_;
};

} // namespace NSolomon::NDataProxy
