package ru.yandex.solomon.dumper;

import java.time.Clock;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executor;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;

import javax.annotation.Nullable;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.memory.layout.MemoryCounter;

/**
 * @author Vladimir Gordiychuk
 */
public class ConcurrentMetricsCache implements MetricsCache {
    private static final Logger logger = LoggerFactory.getLogger(ConcurrentMetricsCache.class);
    private static final long SELF_SIZE = MemoryCounter.objectSelfSizeLayout(ConcurrentMetricsCache.class);
    private static final long EXPIRATION_TIME = TimeUnit.HOURS.toMillis(1L);
    private final Executor executor;
    private final ScheduledExecutorService timer;
    private final Clock clock;
    private final ConcurrentHashMap<Labels, MetricImpl> resolved;
    // TODO: maybe will be better replace on HyperLogLog and reset it state as only cache limit not exceed (gordiychuk@)
    private final ConcurrentHashMap<Labels, AtomicLong> settlers;
    private final ScheduledFuture<?> scheduledTick;
    private final AtomicLong resolvedMemoryUsage = new AtomicLong();
    private final AtomicLong settlersMemoryUsage = new AtomicLong();
    private final MetricsCacheMetrics metrics;
    private volatile ScheduledFuture<?> scheduledRoutine;
    private volatile long latestTick;
    private volatile boolean closed;

    public ConcurrentMetricsCache(Executor executor, ScheduledExecutorService timer, MetricsCacheMetrics metrics) {
        this(executor, timer, Clock.systemUTC(), metrics);
    }

    public ConcurrentMetricsCache(Executor executor, ScheduledExecutorService timer, Clock clock, MetricsCacheMetrics metrics) {
        this.executor = executor;
        this.timer = timer;
        this.clock = clock;
        this.resolved = new ConcurrentHashMap<>();
        this.settlers = new ConcurrentHashMap<>();
        this.metrics = metrics;
        this.scheduledTick = scheduleTick();
        scheduleCleanUpIteration();
    }

    private void tick() {
        this.latestTick = clock.millis();
    }

    public void add(Map<Labels, Metric> update) {
        if (closed) {
            throw new IllegalStateException("Already closed");
        }

        long now = latestTick;
        long bytesAdd = 0;
        int added = 0;
        for (var entry : update.entrySet()) {
            var key = entry.getKey();
            var value = MetricImpl.of(entry.getValue());
            value.touch(now);
            var prev = resolved.put(key, value);
            if (prev == null) {
                bytesAdd += memoryUsage(key);
                bytesAdd += MemoryCounter.HASH_MAP_NODE_SIZE;
                bytesAdd += MetricImpl.SELF_SIZE;
                added++;
            }
        }
        metrics.resolvedAdd.add(added);
        metrics.resolvedSize.add(added);
        resolvedMemoryUsage.addAndGet(bytesAdd);
    }

    public void addSettlers(List<Labels> update) {
        for (Labels labels : update) {
            addSettler(labels);
        }
    }

    public void addSettler(Labels labels) {
        if (closed) {
            throw new IllegalStateException("Already closed");
        }

        var prev = settlers.put(labels, new AtomicLong(latestTick));
        if (prev == null) {
            long bytesAdd = memoryUsage(labels);
            bytesAdd += MemoryCounter.HASH_MAP_NODE_SIZE;
            bytesAdd += MemoryCounter.LONG_SIZE;
            bytesAdd += MemoryCounter.OBJECT_HEADER_SIZE;
            settlersMemoryUsage.addAndGet(bytesAdd);
            metrics.settlersAdd.inc();
            metrics.settlersSize.add(1);
        }
    }

    public void resetSettlers() {
        cleanUpSettlers(Long.MAX_VALUE);
    }

    @Nullable
    @Override
    public Metric resolve(Labels labels) {
        var metric = resolved.get(labels);
        if (metric != null) {
            metrics.resolvedHit.inc();
            metric.touch(clock.millis());
        } else {
            metrics.resolvedMiss.inc();
        }
        return metric;
    }

    @Override
    public boolean containsInSettlers(Labels labels) {
        var result = settlers.get(labels);
        if (result == null) {
            metrics.settlersMiss.inc();
            return false;
        }
        metrics.settlersHit.inc();
        result.set(latestTick);
        return true;
    }

    @Override
    public long memorySizeIncludingSelf() {
        long size = SELF_SIZE;
        size += resolvedMemoryUsage.get();
        size += settlersMemoryUsage.get();
        return size;
    }

    public long resolvedMemorySize() {
        return resolvedMemoryUsage.get();
    }

    public long settlersMemorySize() {
        return settlersMemoryUsage.get();
    }

    @Override
    public void close() {
        closed = true;
        cleanUpResolved(Long.MAX_VALUE);
        cleanUpSettlers(Long.MAX_VALUE);
        scheduledTick.cancel(false);
        var copyScheduledRoutine = scheduledRoutine;
        if (copyScheduledRoutine != null) {
            copyScheduledRoutine.cancel(true);
        }
    }

    private ScheduledFuture<?> scheduleTick() {
        tick();
        long delayMillis = 60_000;
        long jitter = ThreadLocalRandom.current().nextLong(delayMillis);
        return timer.scheduleWithFixedDelay(this::tick, jitter, delayMillis, TimeUnit.MILLISECONDS);
    }

    private void scheduleCleanUpIteration() {
        if (closed) {
            return;
        }

        long jitter = ThreadLocalRandom.current().nextLong(EXPIRATION_TIME / 5);
        scheduledRoutine = timer.schedule(() -> executor.execute(this::cleanUpRoutine), EXPIRATION_TIME + jitter, TimeUnit.MILLISECONDS);
    }

    private void cleanUpRoutine() {
        if (closed) {
            return;
        }

        try {
            long now = clock.millis();
            cleanUpResolved(now);
            cleanUpSettlers(now);
        } catch (Throwable e) {
            logger.error("Cleanup routine failed", e);
        } finally {
            scheduleCleanUpIteration();
        }
    }

    private void cleanUpResolved(long now) {
        long memoryFree = 0;
        int evicted = 0;
        var it = resolved.entrySet().iterator();
        while (it.hasNext()) {
            var entry = it.next();
            var value = entry.getValue();
            if (now - value.getTouchTime() >= EXPIRATION_TIME) {
                it.remove();
                memoryFree += MemoryCounter.HASH_MAP_NODE_SIZE;
                memoryFree += memoryUsage(entry.getKey());
                memoryFree += MetricImpl.SELF_SIZE;
                evicted++;
            }
        }

        if (evicted != 0) {
            metrics.resolvedSize.add(-evicted);
            metrics.resolvedEvict.add(evicted);
            resolvedMemoryUsage.addAndGet(-memoryFree);
        }
    }

    private void cleanUpSettlers(long now) {
        if (settlers.isEmpty()) {
            return;
        }

        long memoryFree = 0;
        int evicted = 0;
        var it = settlers.entrySet().iterator();
        while (it.hasNext()) {
            var entry = it.next();
            var touchedAt = entry.getValue().get();
            if (now - touchedAt >= EXPIRATION_TIME) {
                it.remove();
                memoryFree += MemoryCounter.HASH_MAP_NODE_SIZE;
                memoryFree += memoryUsage(entry.getKey());
                memoryFree += MemoryCounter.LONG_SIZE;
                evicted++;
            }
        }

        if (evicted != 0) {
            metrics.settlersSize.add(-evicted);
            metrics.settlersEvict.add(evicted);
            settlersMemoryUsage.addAndGet(-memoryFree);
        }
    }

    private long memoryUsage(Labels labels) {
        return MemoryCounter.OBJECT_HEADER_SIZE + (MemoryCounter.OBJECT_POINTER_SIZE * labels.size());
    }
}
