package ru.yandex.solomon.dumper;

import java.util.Iterator;
import java.util.NoSuchElementException;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.atomic.AtomicReferenceArray;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import io.grpc.Status;
import org.springframework.context.event.ContextClosedEvent;
import org.springframework.context.event.EventListener;
import org.springframework.core.annotation.Order;

import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.solomon.memory.layout.MemMeasurableSubsystem;
import ru.yandex.solomon.memory.layout.MemoryBySubsystem;

import static java.util.stream.Collectors.toList;

/**
 * @author Vladimir Gordiychuk
 */
public class DumperLocalShards implements Iterable<DumperShard>, MemMeasurableSubsystem {
    private AtomicReferenceArray<DumperShard> shards = new AtomicReferenceArray<>(DumperShardId.SHARD_COUNT);
    private volatile boolean closed;

    @Nullable
    public DumperShard getShardById(int shardId) {
        DumperShard shard = shards.get(idx(shardId));
        if (shard != null && shard.isStop()) {
            if (!closed) {
                remove(shard);
            }
            return null;
        }
        return shard;
    }

    public boolean addShard(DumperShard shard) {
        if (closed) {
            throw Status.ABORTED.withDescription("Process graceful shutdown").asRuntimeException();
        }

        return shards.compareAndSet(idx(shard.getId()), null, shard);
    }

    public boolean remove(DumperShard shard) {
        if (closed) {
            throw Status.ABORTED.withDescription("Process graceful shutdown").asRuntimeException();
        }

        return shards.compareAndSet(idx(shard.getId()), shard, null);
    }

    public Stream<DumperShard> stream() {
        return DumperShardId.idStream()
            .mapToObj(this::getShardById)
            .filter(Objects::nonNull);
    }

    private static int idx(int shardId) {
        return shardId - DumperShardId.FIRST_SHARD_ID;
    }

    @Override
    @Nonnull
    public Iterator<DumperShard> iterator() {
        return new Iterator<>() {
            private int index = DumperShardId.FIRST_SHARD_ID;
            private DumperShard next;

            @Override
            public boolean hasNext() {
                while (index <= DumperShardId.LAST_SHARD_ID) {
                    next = getShardById(index);
                    index++;
                    if (next != null) {
                        return true;
                    }
                }
                return false;
            }

            @Override
            public DumperShard next() {
                if (next == null) {
                    throw new NoSuchElementException();
                }
                return next;
            }
        };
    }

    @Order(1)
    @EventListener(ContextClosedEvent.class)
    public void gracefulShutdownSync() {
        // https://jira.spring.io/browse/SPR-17298
        gracefulShutdown().join();
    }

    public CompletableFuture<Void> gracefulShutdown() {
        closed = true;
        return stream()
            .map(shard -> {
                shards.compareAndSet(idx(shard.getId()), shard, null);
                return shard.stop();
            })
            .collect(Collectors.collectingAndThen(toList(), CompletableFutures::allOfVoid));
    }

    @Override
    public void addMemoryBySubsystem(MemoryBySubsystem memory) {
        stream().forEach(dumperShard -> dumperShard.addMemoryBySubsystem(memory));
    }
}
