package ru.yandex.solomon.dumper;

import java.util.EnumMap;
import java.util.List;
import java.util.Map;
import java.util.function.BiConsumer;

import ru.yandex.monlib.metrics.MetricConsumer;
import ru.yandex.monlib.metrics.MetricSupplier;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.meter.Meter;
import ru.yandex.monlib.metrics.primitives.GaugeInt64;
import ru.yandex.monlib.metrics.primitives.Rate;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.dumper.storage.shortterm.file.FileName;
import ru.yandex.solomon.dumper.storage.shortterm.file.FileNamesList;
import ru.yandex.solomon.dumper.storage.shortterm.file.FileType;

/**
 * @author Vladimir Gordiychuk
 */
public class DumperShardMetrics implements MetricSupplier {
    public final DumperShardMetricsAggregated aggregated;
    private final MetricRegistry registry;
    public final GaugeInt64 readLagSec;
    public final GaugeInt64 writeLagSec;
    public final GaugeInt64 pendingTxn;
    public final Rate preparedTxn;
    public final Rate committedTxn;
    public final Rate readBytes;
    public final Rate readFiles;
    public final Rate kvErrors;
    public volatile long lastErrorInstant = 0;
    public final Map<FileType, FileMetrics> fileMetrics;
    public final Meter cpuTimeNanos;
    private final int metricsCount;
    public final long createdAtMs = System.currentTimeMillis();

    public DumperShardMetrics(String shardId, DumperShardMetricsAggregated aggregated) {
        this.aggregated = aggregated;
        this.registry = new MetricRegistry(Labels.of("shardId", shardId));
        this.readLagSec = registry.gaugeInt64("dumper.shard.txn.readLagSec");
        this.writeLagSec = registry.gaugeInt64("dumper.shard.txn.writeLagSec");
        this.pendingTxn = registry.gaugeInt64("dumper.shard.txn.pending");
        this.preparedTxn = registry.rate("dumper.shard.txn.prepared");
        this.committedTxn = registry.rate("dumper.shard.txn.committed");
        this.readBytes = registry.rate("dumper.shard.txn.read.bytes");
        this.readFiles = registry.rate("dumper.shard.txn.read.files");
        this.kvErrors = registry.rate("dumper.shard.kv.errors");
        this.cpuTimeNanos = registry.fiveMinutesMeter("dumper.shard.cpuTimeNanos");
        var fileMetrics = new EnumMap<FileType, FileMetrics>(FileType.class);
        for (FileType type : FileType.values()) {
            fileMetrics.put(type, new FileMetrics(registry.subRegistry("type", type.name())));
        }
        this.fileMetrics = fileMetrics;
        this.metricsCount = registry.estimateCount() + 2;
    }

    @Override
    public int estimateCount() {
        return metricsCount;
    }

    public void readBytes(long bytes) {
        readBytes.add(bytes);
        aggregated.readBytes.add(bytes);
        aggregated.readBytesSize.record(bytes);
    }

    public void readFiles(long count) {
        readFiles.add(count);
        aggregated.readFiles.add(count);
    }

    public void prepareTxn(long count) {
        pendingTxn.add(count);
        preparedTxn.add(count);
        aggregated.preparedTxn.add(count);
    }

    public void kvError() {
        kvErrors.inc();
        aggregated.kvErrors.inc();
        lastErrorInstant = System.currentTimeMillis();
    }

    public void commitTxn(long count) {
        pendingTxn.add(-count);
        committedTxn.add(count);
        aggregated.committedTxn.add(count);
    }

    @Override
    public void append(long tsMillis, Labels commonLabels, MetricConsumer consumer) {
        registry.append(tsMillis, commonLabels, consumer);

        // total files
        var labels = this.registry.getCommonLabels().add("type", "total");
        var registry = new MetricRegistry(labels);
        var totalFiles = new FileMetrics(registry);
        fileMetrics.values().forEach(totalFiles::combine);
        registry.append(tsMillis, commonLabels, consumer);
    }

    public void add(FileType type, int bytes, int count) {
        fileMetrics.get(type).add(bytes, count);
    }

    public void set(FileNamesList files) {
        update(files, FileMetrics::set);
    }

    public void add(FileNamesList files) {
        update(files, FileMetrics::add);
    }

    public void update(FileNamesList files, BiConsumer<FileMetrics, SizeAndCount> fn) {
        for (FileType type : FileType.values()) {
            var sizeAndCount = SizeAndCount.of(files.list(type), false);
            var metrics = fileMetrics.get(type);
            fn.accept(metrics, sizeAndCount);
        }
    }

    public void change(FileType from, FileType to, FileNamesList files) {
        change(from, to, files.list(from));
    }

    public void change(FileType from, FileType to, List<? extends FileName> files) {
        var sizeAndCount = SizeAndCount.of(files, true);
        change(from, to, sizeAndCount.bytes, sizeAndCount.count);
    }

    public void change(FileType from, FileType to, long bytes, long count) {
        fileMetrics.get(from).add(-bytes, -count);
        fileMetrics.get(to).add(bytes, count);
    }

    public static class FileMetrics {
        public final GaugeInt64 count;
        public final GaugeInt64 bytes;

        public FileMetrics(MetricRegistry registry) {
            this.count = registry.gaugeInt64("dumper.shard.kv.files.count");
            this.bytes = registry.gaugeInt64("dumper.shard.kv.files.bytes");
        }

        public void add(SizeAndCount sizeAndCount) {
            add(sizeAndCount.bytes, sizeAndCount.count);
        }

        public void set(SizeAndCount sizeAndCount) {
            this.bytes.set(sizeAndCount.bytes);
            this.count.set(sizeAndCount.count);
        }

        public void add(long bytes, long count) {
            this.bytes.add(bytes);
            this.count.add(count);
        }

        public void combine(FileMetrics metrics) {
            this.bytes.combine(metrics.bytes);
            this.count.combine(metrics.count);
        }
    }

    private static class SizeAndCount {
        private static final SizeAndCount EMPTY = new SizeAndCount(0, 0);
        private final int bytes;
        private final int count;

        public SizeAndCount(int bytes, int count) {
            this.bytes = bytes;
            this.count = count;
        }

        public static SizeAndCount of(List<? extends FileName> files, boolean ignoreInvalid) {
            if (files.isEmpty()) {
                return EMPTY;
            }

            int bytes = 0;
            int count = 0;
            for (var file : files) {
                if (ignoreInvalid && file.isValid()) {
                    continue;
                }
                bytes += file.getBytesSize();
                count += file.getFileCount();
            }
            return new SizeAndCount(bytes, count);
        }
    }
}
