package ru.yandex.solomon.dumper;

import java.util.Map;

import ru.yandex.misc.lang.EnumMapUtils;
import ru.yandex.monlib.metrics.MetricConsumer;
import ru.yandex.monlib.metrics.MetricSupplier;
import ru.yandex.monlib.metrics.histogram.HistogramCollector;
import ru.yandex.monlib.metrics.histogram.Histograms;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.dumper.mon.MetricsSupplierUtils;
import ru.yandex.solomon.dumper.storage.shortterm.file.FileType;

/**
 * @author Vladimir Gordiychuk
 */
public class DumperShardMetricsInstant implements MetricSupplier {
    private HistogramCollector readLagSec = Histograms.exponential(20, 2);
    private HistogramCollector writeLagSec = Histograms.exponential(20, 2);
    private long pendingTxn = 0;
    private Map<FileType, FileStat> fileMetrics = EnumMapUtils.fill(FileType.class, type -> new FileStat());
    private int shardCount;

    public void combine(DumperShardMetrics metrics) {
        shardCount++;
        readLagSec.collect(metrics.readLagSec.get());
        writeLagSec.collect(metrics.writeLagSec.get());
        pendingTxn += metrics.pendingTxn.get();
        for (FileType type : FileType.values()) {
            var source = metrics.fileMetrics.get(type);
            var target = fileMetrics.get(type);
            target.combine(source);
        }
    }

    @Override
    public int estimateCount() {
        return 5 + fileMetrics.size() * 2;
    }

    @Override
    public void append(long tsMillis, Labels commonLabels, MetricConsumer consumer) {
        MetricsSupplierUtils.append("dumper.host.txn.readLagSec", readLagSec.snapshot(), consumer);
        MetricsSupplierUtils.append("dumper.host.txn.writeLagSec", writeLagSec.snapshot(), consumer);
        MetricsSupplierUtils.append("dumper.host.txn.pending", pendingTxn, consumer);
        MetricsSupplierUtils.append("dumper.host.shards.count", shardCount, consumer);
        appendFilesStats(consumer);
    }

    private void appendFilesStats(MetricConsumer consumer) {
        FileStat total = new FileStat();
        for (FileType type : FileType.values()) {
            var stats = fileMetrics.get(type);
            total.combine(stats);
            stats.append(Labels.of("type", type.name()), consumer);
        }
        total.append(Labels.of("type", "total"), consumer);
    }

    private static class FileStat {
        long bytes;
        long count;

        public void combine(DumperShardMetrics.FileMetrics metrics) {
            bytes += metrics.bytes.get();
            count += metrics.count.get();
        }

        public void combine(FileStat stat) {
            bytes += stat.bytes;
            count += stat.count;
        }

        public void append(Labels commonLabels, MetricConsumer consumer) {
            MetricsSupplierUtils.append("dumper.host.kv.files.count", commonLabels, count, consumer);
            MetricsSupplierUtils.append("dumper.host.kv.files.bytes", commonLabels, bytes, consumer);
        }
    }
}
