package ru.yandex.solomon.dumper;

import javax.annotation.Nullable;

import com.google.common.base.Throwables;
import io.netty.buffer.ByteBufAllocator;
import io.netty.buffer.ByteBufUtil;
import it.unimi.dsi.fastutil.ints.Int2ObjectMap;
import it.unimi.dsi.fastutil.ints.Int2ObjectOpenHashMap;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.misc.thread.WhatThreadDoes;
import ru.yandex.monlib.metrics.MetricType;
import ru.yandex.monlib.metrics.labels.LabelAllocator;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.dumper.storage.shortterm.DumperTx;
import ru.yandex.solomon.dumper.storage.shortterm.ProducerKey;
import ru.yandex.solomon.slog.Log;
import ru.yandex.solomon.slog.ResolvedLogMetaBuilder;
import ru.yandex.solomon.slog.ResolvedLogMetaBuilderImpl;
import ru.yandex.solomon.slog.ResolvedLogMetaHeader;
import ru.yandex.solomon.slog.SnapshotLogDataHeader;
import ru.yandex.solomon.slog.UnresolvedLogMetaBuilder;
import ru.yandex.solomon.slog.UnresolvedLogMetaBuilderImpl;
import ru.yandex.solomon.slog.UnresolvedLogMetaHeader;
import ru.yandex.solomon.slog.UnresolvedLogMetaIteratorImpl;
import ru.yandex.solomon.slog.UnresolvedLogMetaRecord;
import ru.yandex.solomon.slog.compression.DecodeStream;
import ru.yandex.solomon.slog.compression.EncodeStream;
import ru.yandex.stockpile.api.EDecimPolicy;

/**
 * @author Vladimir Gordiychuk
 */
public class ParsingTask {
    private static final Logger logger = LoggerFactory.getLogger(ParsingTask.class);
    private final DumperTx tx;
    // TODO: move it to stockpile, that will resolve by numId from archive correspond decimPolicy
    private final EDecimPolicy decimPolicy;
    private final Log log;
    private final LabelAllocator labelAllocator;
    private final ByteBufAllocator allocator;
    private final MetricsCache cache;
    private final Int2ObjectOpenHashMap<Request> requests = new Int2ObjectOpenHashMap<>();
    @Nullable
    private Unresolved unresolved;

    private int prevShardId = 0;
    private Request prevRequest;
    private int byteSize;

    private ParsingStats stats;
    private long createdAt;

    public ParsingTask(DumperTx tx, EDecimPolicy decimPolicy, Log log, MetricsCache cache, LabelAllocator labelAllocator, ByteBufAllocator allocator) {
        this.tx = tx;
        this.decimPolicy = decimPolicy;
        this.log = log;
        this.cache = cache;
        this.labelAllocator = labelAllocator;
        this.allocator = allocator;
        this.stats = new ParsingStats();
        this.createdAt = System.nanoTime();
    }

    Log getLog() {
        return log;
    }

    public ParsingResult run() {
        long startedAt = System.nanoTime();
        stats.waitingTimeNs = startedAt - createdAt;
        var h = WhatThreadDoes.push("Parsing shard " + Integer.toUnsignedLong(log.numId));
        try {
            return parse();
        } catch (Throwable e) {
            logCorruptedData(e);
            requests.values().forEach(Request::close);
            Throwables.throwIfUnchecked(e);
            throw new RuntimeException(e);
        } finally {
            log.close();
            if (unresolved != null) {
                unresolved.close();
            }
            h.popSafely();
            stats.parsingTimeNs = System.nanoTime() - startedAt;
        }
    }

    private void logCorruptedData(Throwable e) {
        if (Throwables.getRootCause(e) instanceof OutOfMemoryError) {
            return;
        }

        logger.warn("Corruption at " +
            tx +
            " for numId: " +
            Integer.toUnsignedLong(log.numId) +
            "\nMeta:\n" +
            ByteBufUtil.prettyHexDump(log.meta, 0, log.meta.writerIndex()) +
            "\nData:\n" +
            ByteBufUtil.prettyHexDump(log.data, 0, log.data.writerIndex()), e);
    }

    private ParsingResult parse() {
        UnresolvedLogMetaHeader metaHeader = new UnresolvedLogMetaHeader(log.meta);
        SnapshotLogDataHeader dataHeader = new SnapshotLogDataHeader(log.data);
        UnresolvedLogMetaRecord metaRecord = new UnresolvedLogMetaRecord();
        try (var metaIt = new UnresolvedLogMetaIteratorImpl(metaHeader, log.meta, labelAllocator);
             var dataStream = DecodeStream.create(dataHeader.compressionAlg, log.data.retain()))
        {
            while (metaIt.next(metaRecord)) {
                var metric = cache.resolve(metaRecord.labels);
                if (metric == null) {
                    flush(dataStream);
                    reset();
                    if (!cache.containsInSettlers(metaRecord.labels)) {
                        addUnresolved(metaHeader, dataHeader, dataStream, metaRecord);
                    } else {
                        stats.metricsDropped++;
                        stats.pointsDropped += metaRecord.points;
                        dataStream.skipBytes(metaRecord.dataSize);
                    }
                    continue;
                } else if (metric.getType() != metaRecord.type) {
                    flush(dataStream);
                    reset();
                    addUnresolved(metaHeader, dataHeader, dataStream, metaRecord);
                    continue;
                }

                if (prevShardId == metric.getShardId()) {
                    prevRequest.addMeta(metric.getType(), metric.getLocalId(), metaRecord.points, metaRecord.dataSize);
                    byteSize += metaRecord.dataSize;
                    continue;
                }

                flush(dataStream);
                prevShardId = metric.getShardId();
                prevRequest = getOrCreateRequest(metric.getShardId(), dataHeader);
                prevRequest.addMeta(metric.getType(), metric.getLocalId(), metaRecord.points, metaRecord.dataSize);
                byteSize = metaRecord.dataSize;
            }
            flush(dataStream);
            reset();
            return new ParsingResult(prepareRequests(dataHeader), prepareUnresolved(), stats);
        }
    }

    private Int2ObjectMap<Log> prepareRequests(SnapshotLogDataHeader header) {
        var result = new Int2ObjectOpenHashMap<Log>(requests.size());
        try {
            var it = requests.int2ObjectEntrySet().fastIterator();
            while (it.hasNext()) {
                var entry = it.next();
                var shardId = entry.getIntKey();
                var request = entry.getValue();

                stats.metricsKnown += request.metrics;
                stats.pointsKnown += request.points;
                var log = request.build(header);
                result.put(shardId, log);

                it.remove();
                request.close();
            }
            return result;
        } catch (Throwable e) {
            result.values().forEach(Log::close);
            Throwables.throwIfUnchecked(e);
            throw new RuntimeException(e);
        }
    }

    @Nullable
    private Log prepareUnresolved() {
        if (unresolved == null) {
            return null;
        }

        stats.metricsUnknown = unresolved.metrics;
        stats.pointsUnknown = unresolved.points;
        return unresolved.build();
    }

    private Request getOrCreateRequest(int shardId, SnapshotLogDataHeader header) {
        var result = requests.get(shardId);
        if (result == null) {
            result = new Request(tx, decimPolicy, header, allocator);
            requests.put(shardId, result);
        }
        return result;
    }

    private void addUnresolved(UnresolvedLogMetaHeader metaHeader, SnapshotLogDataHeader dataHeader, DecodeStream dataStream, UnresolvedLogMetaRecord record) {
        if (unresolved == null) {
            unresolved = new Unresolved(metaHeader, dataHeader, allocator);
            unresolved.metaBuilder.onCommonLabels(Labels.of());
        }
        unresolved.metaBuilder.onMetric(record.type, record.labels, record.points, record.dataSize);
        copy(dataStream, unresolved.dataEncoder, record.dataSize);
        unresolved.points += record.points;
        unresolved.metrics += 1;
    }

    private void reset() {
        prevShardId = 0;
        prevRequest = null;
        byteSize = 0;
    }

    private void flush(DecodeStream source) {
        if (prevRequest == null) {
            return;
        }

        copy(source, prevRequest.dataEncoder, byteSize);
    }

    private void copy(DecodeStream src, EncodeStream dst, int byteSize) {
        dst.write(src, byteSize);
    }

    private static class Request implements AutoCloseable {
        private ResolvedLogMetaBuilder metaBuilder;
        private EncodeStream dataEncoder;
        private int points;
        private int metrics;

        public Request(DumperTx tx, EDecimPolicy decimPolicy, SnapshotLogDataHeader header, ByteBufAllocator allocator) {
            var metaHeader = new ResolvedLogMetaHeader(header.numId, header.compressionAlg)
                    .setProducerId(ProducerKey.makeProducerId(tx))
                    .setProducerSeqNo(tx.getTxn())
                    .setDecimPolicy(decimPolicy);

            try {
                this.metaBuilder = new ResolvedLogMetaBuilderImpl(metaHeader, allocator);
                this.dataEncoder = EncodeStream.create(header.compressionAlg, allocator);
                this.dataEncoder.writeHeader(buf -> buf.writeZero(header.size()));
            } catch (Throwable e) {
                if (metaBuilder != null) {
                    metaBuilder.close();
                }
                if (dataEncoder != null) {
                    dataEncoder.close();
                }
                throw new RuntimeException(e);
            }
        }

        public void addMeta(MetricType type, long localId, int points, int dataSize) {
            this.metaBuilder.onMetric(type, localId, points, dataSize);
            this.metrics += 1;
            this.points += points;
        }

        public Log build(SnapshotLogDataHeader header) {
            var data = dataEncoder.finish();
            try {
                int pos = data.writerIndex();
                header.writeTo(data.resetWriterIndex(), metrics, points);
                data.writerIndex(pos);
                var meta = metaBuilder.build();
                return new Log(header.numId, meta, data);
            } catch (Throwable e) {
                data.release();
                Throwables.throwIfUnchecked(e);
                throw new RuntimeException(e);
            }
        }

        @Override
        public void close() {
            metaBuilder.close();
            dataEncoder.close();
        }
    }

    private static class Unresolved implements AutoCloseable {
        private final UnresolvedLogMetaBuilder metaBuilder;
        private final SnapshotLogDataHeader dataHeader;
        private final EncodeStream dataEncoder;
        private int points;
        private int metrics;

        public Unresolved(UnresolvedLogMetaHeader metaHeader, SnapshotLogDataHeader dataHeader, ByteBufAllocator allocator) {
            this.metaBuilder = new UnresolvedLogMetaBuilderImpl(metaHeader.numId, metaHeader.compressionAlg, allocator);
            this.dataHeader = dataHeader;
            this.dataEncoder = EncodeStream.create(dataHeader.compressionAlg, allocator);
            this.dataEncoder.writeHeader(buf -> buf.writeZero(dataHeader.size()));
        }

        public Log build() {
            var data = dataEncoder.finish();
            try {
                int pos = data.writerIndex();
                dataHeader.writeTo(data.resetWriterIndex(), metrics, points);
                data.writerIndex(pos);
                return new Log(dataHeader.numId, metaBuilder.build(), data);
            } catch (Throwable e) {
                data.release();
                Throwables.throwIfUnchecked(e);
                throw new RuntimeException(e);
            }
        }

        @Override
        public void close() {
            metaBuilder.close();
            dataEncoder.close();
        }
    }
}
