package ru.yandex.solomon.dumper;

import ru.yandex.monlib.metrics.MetricConsumer;
import ru.yandex.monlib.metrics.MetricSupplier;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.meter.Meter;
import ru.yandex.monlib.metrics.primitives.GaugeInt64;
import ru.yandex.monlib.metrics.primitives.LazyRate;
import ru.yandex.monlib.metrics.primitives.Rate;
import ru.yandex.monlib.metrics.registry.MetricRegistry;

/**
 * @author Vladimir Gordiychuk
 */
public class SolomonShardMetrics implements MetricSupplier {
    private final MetricRegistry registry;
    public final Rate metricsKnown;
    public final Rate metricsUnknown;
    public final Rate metricsDropped;
    public final LazyRate metricsTotal;

    public final Rate pointsKnown;
    public final Rate pointsUnknown;
    public final Rate pointsDropped;
    public final LazyRate pointsTotal;

    public final Meter cpuTimeNanos;
    public final Rate parsingSuccess;
    public final Rate parsingError;
    public final Rate resolveSuccess;
    public final Rate resolveError;
    public final Rate writeSuccess;
    public final Rate writeError;

    public final GaugeInt64 txLagSec;

    public SolomonShardMetrics() {
        this.registry = new MetricRegistry();
        this.metricsKnown = registry.rate("dumper.log.metrics", Labels.of("type", "known"));
        this.metricsUnknown = registry.rate("dumper.log.metrics", Labels.of("type", "unknown"));
        this.metricsDropped = registry.rate("dumper.log.metrics", Labels.of("type", "dropped"));
        this.metricsTotal = registry.lazyRate("dumper.log.metrics", Labels.of("type", "total"), () -> metricsKnown.get() + metricsUnknown.get() + metricsDropped.get());

        this.pointsKnown = registry.rate("dumper.log.points", Labels.of("type", "known"));
        this.pointsUnknown = registry.rate("dumper.log.points", Labels.of("type", "unknown"));
        this.pointsDropped = registry.rate("dumper.log.points", Labels.of("type", "dropped"));
        this.pointsTotal = registry.lazyRate("dumper.log.points", Labels.of("type", "total"), () -> pointsKnown.get() + pointsUnknown.get() + pointsDropped.get());
        this.cpuTimeNanos = registry.fiveMinutesMeter("dumper.cpuTimeNanos");

        this.parsingSuccess = registry.rate("dumper.parsing.completedOk");
        this.parsingError = registry.rate("dumper.parsing.completedError");
        this.resolveSuccess = registry.rate("dumper.log.resolve.completedOk");
        this.resolveError = registry.rate("dumper.log.resolve.completedError");
        this.writeSuccess = registry.rate("dumper.log.write.completedOk");
        this.writeError = registry.rate("dumper.log.write.completedError");
        this.txLagSec = registry.gaugeInt64("dumper.log.txLagSec");
    }

    @Override
    public int estimateCount() {
        return registry.estimateCount();
    }

    public void successParse(ParsingStats stats) {
        metricsKnown.add(stats.metricsKnown);
        metricsKnown.add(stats.metricsKnown);
        metricsUnknown.add(stats.metricsUnknown);
        metricsDropped.add(stats.metricsDropped);
        pointsKnown.add(stats.pointsKnown);
        pointsUnknown.add(stats.pointsUnknown);
        pointsDropped.add(stats.pointsDropped);
        parsingSuccess.inc();
    }

    public void failedParse() {
        parsingError.inc();
    }

    public void combine(SolomonShardMetrics metrics) {
        this.metricsKnown.combine(metrics.metricsKnown);
        this.metricsUnknown.combine(metrics.metricsUnknown);
        this.metricsDropped.combine(metrics.metricsDropped);

        this.pointsKnown.combine(metrics.pointsKnown);
        this.pointsUnknown.combine(metrics.pointsUnknown);
        this.pointsDropped.combine(metrics.pointsDropped);

        this.cpuTimeNanos.combine(metrics.cpuTimeNanos);
        this.parsingSuccess.combine(metrics.parsingSuccess);
        this.parsingError.combine(metrics.parsingError);

        this.parsingSuccess.combine(metrics.parsingSuccess);
        this.parsingError.combine(metrics.parsingError);
        this.resolveSuccess.combine(metrics.resolveSuccess);
        this.resolveError.combine(metrics.resolveError);
        this.writeSuccess.combine(metrics.writeSuccess);
        this.writeError.combine(metrics.writeError);

        this.txLagSec.combine(metrics.txLagSec);
    }

    @Override
    public void append(long tsMillis, Labels commonLabels, MetricConsumer consumer) {
        registry.append(tsMillis, commonLabels, consumer);
    }
}
